#!/usr/bin/env bash
#
# rotate_monthly.sh – swap the old RSA‑4096 (or Ed25519) vault for a fresh one
# -------------------------------------------------------------------------
# What it does
#   1. Generates a new monthly key pair (RSA‑4096 by default, switch to ed25519 if you prefer).  
#   2. Exports the new public key to the web‑folder so visitors can verify signatures.  
#   3. Re‑assembles the vault room with three newly‑picked OTP fragments.  
#   4. Moves the previous vault (and its public key) to an archive sub‑folder.  
#   5. Updates a simple CHANGELOG.txt entry so you have a record of the rotation.  
#
# Prerequisites
#   • generate_keys.sh has already created the first monthly key (stored in your GPG keyring).  
#   • make_fragments.sh has a pool of random fragments in web/honeypot/fragments/.  
#   • build_snowflake.sh has placed the old vault (vault_room.asc) in web/honeypot/.  
#
# Run this once at the start of each new month (or whenever you want to refresh the high‑value vault).  
# -------------------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# -------------------- 1. Configuration --------------------
BASE_DIR="$(dirname "$(realpath "$0")")/.."          # project root
FRAG_DIR="${BASE_DIR}/web/honeypot/fragments"
OUT_DIR="${BASE_DIR}/web/honeypot"
ARCHIVE_DIR="${OUT_DIR}/archived_vaults"
TMP_DIR="$(mktemp -d)"                               # temporary workspace

# How many fragments the vault should require (usually 3)
VAULT_FRAG_COUNT=3

# -------------------- 2. Archive the old vault --------------------
mkdir -p "$ARCHIVE_DIR"

if [[ -f "${OUT_DIR}/vault_room.asc" ]]; then
    TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
    mv "${OUT_DIR}/vault_room.asc" "${ARCHIVE_DIR}/vault_room_${TIMESTAMP}.asc"
    echo "Archived previous vault as vault_room_${TIMESTAMP}.asc"
fi

# Also archive the old monthly public key (if it exists)
if [[ -f "${OUT_DIR}/keys/monthly_pub.asc" ]]; then
    mv "${OUT_DIR}/keys/monthly_pub.asc" "${ARCHIVE_DIR}/monthly_pub_${TIMESTAMP}.asc"
    echo "Archived previous monthly public key."
fi

# -------------------- 3. Create a fresh monthly key --------------------
# Prompt for a strong passphrase (won’t echo on the screen)
echo "=== Generating a new monthly RSA‑4096 key – please use a strong, random passphrase ==="
read -rsp "Enter passphrase: " MONTHLY_PW
echo

# Build a temporary batch file for GPG
cat > /tmp/gpg_monthly_batch <<EOF
%echo Generating new monthly RSA‑4096 key
Key-Type: rsa
Key-Length: 4096
Subkey-Type: rsa
Subkey-Length: 4096
Name-Real: Honeypot Monthly 2026
Expire-Date: 0
Passphrase: $MONTHLY_PW
%commit
%echo done
EOF

gpg --batch --gen-key /tmp/gpg_monthly_batch
rm -f /tmp/gpg_monthly_batch

# Export the new public key for the web folder
gpg --armor --export "Honeypot Monthly 2026" > "${OUT_DIR}/keys/monthly_pub.asc"
echo "New monthly public key written to ${OUT_DIR}/keys/monthly_pub.asc"

# -------------------- 4. Pick fresh fragments for the new vault --------------------
echo "Selecting $VAULT_FRAG_COUNT fresh fragments for the new vault…"
SELECTED_FRAGS=()
while [[ ${#SELECTED_FRAGS[@]} -lt $VAULT_FRAG_COUNT ]]; do
    CANDIDATE=$(ls "$FRAG_DIR"/frag_*.bin | shuf -n1)
    # Avoid duplicates
    if [[ ! " ${SELECTED_FRAGS[*]} " =~ " $CANDIDATE " ]]; then
        SELECTED_FRAGS+=("$CANDIDATE")
    fi
done

# Write a short markdown note that explains the vault’s dependencies
VAULT_NOTE="${TMP_DIR}/vault_room.md"
cat > "$VAULT_NOTE" <<EOF
# Vault Room (High‑Value)

To open this vault you need the following three OTP fragments:

$(for f in "${SELECTED_FRAGS[@]}"; do echo "- $(basename "$f")"; done)

When all three are provided, the archive reveals a random hash that serves as a dead‑end token. Good luck.
EOF

# Assemble the vault tarball
VAULT_BUNDLE="${TMP_DIR}/vault_room.tar.gz"
tar czf "$VAULT_BUNDLE" -C "$TMP_DIR" "$(basename "$VAULT_NOTE")"
for f in "${SELECTED_FRAGS[@]}"; do
    tar --append -f "$VAULT_BUNDLE" -C "$(dirname "$f")" "$(basename "$f")"
done
gzip -f "$VAULT_BUNDLE"   # ensure it’s a .tar.gz

# Encrypt with the freshly generated monthly key
gpg --armor --encrypt \
    --recipient "Honeypot Monthly 2026" \
    --output "${OUT_DIR}/vault_room.asc" "${VAULT_BUNDLE%.gz}"

echo "New vault created: ${OUT_DIR}/vault_room.asc"

# -------------------- 5. Update CHANGELOG --------------------
CHANGELOG="${BASE_DIR}/CHANGELOG.txt"
{
    echo ""
    echo "=== $(date +"%Y-%m-%d") Rotation ==="
    echo "* New monthly RSA‑4096 key generated."
    echo "* Vault rebuilt with fresh fragments: $(basename "${SELECTED_FRAGS[0]}"), $(basename "${SELECTED_FRAGS[1]}"), $(basename "${SELECTED_FRAGS[2]}")."
    echo "* Old vault and public key archived under ${ARCHIVE_DIR}."
} >> "$CHANGELOG"

echo "CHANGELOG updated at $CHANGELOG"

# -------------------- 6. Clean up --------------------
rm -rf "$TMP_DIR"

echo ""
echo "Monthly rotation complete. Upload the updated ${OUT_DIR} directory to your web server (if you’re working locally it’s already there)."