#!/usr/bin/env bash
#
# make_fragments.sh – generate a pool of random OTP fragments
# ---------------------------------------------------------
# Each fragment is a binary file (16‑32 bytes) that will be
# referenced by the conversation bundles and the snowflake rooms.
# The script writes the fragments into a folder that the other
# scripts can read from.
#
# Run this after you have generated your GPG keys.
# ----------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# ---------- 1. Configuration ----------
FRAG_COUNT=12               # total number of fragments you want
MIN_SIZE=16                 # smallest fragment size (bytes)
MAX_SIZE=32                 # largest fragment size (bytes)

# Output directory – sibling of the web folder so it’s easy to copy later
OUT_DIR="$(dirname "$(realpath "$0")")/../web/honeypot/fragments"
mkdir -p "$OUT_DIR"

# ---------- 2. Generate fragments ----------
echo "Generating $FRAG_COUNT random OTP fragments (sizes $MIN_SIZE‑$MAX_SIZE bytes)…"

for i in $(seq 1 "$FRAG_COUNT"); do
    # Pick a random size between MIN_SIZE and MAX_SIZE
    SIZE=$(shuf -i "$MIN_SIZE"-"$MAX_SIZE" -n 1)

    # Name the file frag_01.bin, frag_02.bin, …
    FILENAME=$(printf "frag_%02d.bin" "$i")

    # Create the random binary file
    dd if=/dev/urandom bs="$SIZE" count=1 of="$OUT_DIR/$FILENAME" status=none

    echo "  → $FILENAME ($SIZE bytes)"
done

echo ""
echo "All fragments are now in:"
echo "  $OUT_DIR"
echo "You can reference them in the later bundle‑creation scripts."