#!/usr/bin/env bash
#
# geoip_gate.sh – enable geographic restrictions for /honeypot/
# ----------------------------------------------------------------------
# What it does
#   1. Installs the free GeoLite2‑Country database (requires internet access).
#   2. Enables the GeoIP2 module for nginx or the MaxMindDB module for Apache.
#   3. Writes a small config snippet that allows only the countries you list.
#   4. Reloads the web server so the rule takes effect immediately.
#
# Prerequisites
#   • You already have nginx or Apache serving the /honeypot/ directory.
#   • You have sudo privileges on your Linux workstation.
#
# Usage
#   sudo ./scripts/geoip_gate.sh US CA   # allow United States and Canada
#   sudo ./scripts/geoip_gate.sh ALL      # disable gating (open to everyone)
# ----------------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# -------------------- 1. Parse arguments --------------------
if [[ $# -eq 0 ]]; then
    echo "Usage: sudo $0 <COUNTRY_CODE> [<COUNTRY_CODE> …]  (e.g. US CA) or ALL"
    exit 1
fi

# If the user passes "ALL", we will skip creating any restriction.
if [[ "$1" == "ALL" ]]; then
    ALLOWED_COUNTRIES=".*"   # match anything
else
    # Build a regex like (US|CA|GB)
    ALLOWED_COUNTRIES="^($(printf '%s|' "$@" | sed 's/|$//'))$"
fi

# -------------------- 2. Install GeoIP2 database --------------------
echo "Installing GeoLite2‑Country database..."
sudo apt-get update -qq
sudo apt-get install -y -qq libmaxminddb0 libmaxminddb-dev mmdb-bin wget

# Download the latest free database (updated monthly)
TMP_DB="/tmp/GeoLite2-Country.mmdb"
wget -q -O "$TMP_DB" "https://geolite.maxmind.com/download/geoip/database/GeoLite2-Country.mmdb"
sudo mkdir -p /usr/share/GeoIP
sudo mv "$TMP_DB" /usr/share/GeoIP/GeoLite2-Country.mmdb
echo "Database installed at /usr/share/GeoIP/GeoLite2-Country.mmdb"

# -------------------- 3. Detect which web server is running --------------------
if command -v nginx >/dev/null 2>&1; then
    SERVER="nginx"
elif command -v apache2ctl >/dev/null 2>&1; then
    SERVER="apache"
else
    echo "Neither nginx nor Apache detected. Install one of them first."
    exit 1
fi

# -------------------- 4. Write the restriction config --------------------
if [[ "$SERVER" == "nginx" ]]; then
    # Ensure the GeoIP2 module is loaded (most nginx packages include it)
    NGX_CONF="/etc/nginx/conf.d/honeypot_geoip.conf"
    sudo tee "$NGX_CONF" > /dev/null <<EOF
# GeoIP2 country restriction for /honeypot/
geoip2 /usr/share/GeoIP/GeoLite2-Country.mmdb {
    \$geoip2_country_code default=ZZ source=\$remote_addr;
}
server {
    # Assuming you already have a server block for your domain.
    # This location block only affects the /honeypot/ path.
    location /honeypot/ {
        # Allow only the countries matched by the regex above.
        if (\$geoip2_country_code !~ $ALLOWED_COUNTRIES) {
            return 403 "Access denied – your country is not authorized.";
        }
        # Normal serving – the alias points to the actual folder.
        alias /var/www/html/honeypot/;
        autoindex on;
    }
}
EOF
    echo "nginx GeoIP restriction written to $NGX_CONF"
    sudo nginx -t && sudo systemctl reload nginx
    echo "nginx configuration reloaded."

elif [[ "$SERVER" == "apache" ]]; then
    # Enable the MaxMindDB module if not already enabled
    sudo a2enmod maxminddb >/dev/null 2>&1 || true

    APACHE_CONF="/etc/apache2/conf-available/honeypot_geoip.conf"
    sudo tee "$APACHE_CONF" > /dev/null <<EOF
# MaxMindDB country restriction for /honeypot/
MaxMindDBFile DB /usr/share/GeoIP/GeoLite2-Country.mmdb
MaxMindDBEnv COUNTRY_CODE DB/country/iso_code

<Directory "/var/www/html/honeypot">
    # Allow only the specified countries
    <RequireAll>
        Require env COUNTRY_CODE ${ALLOWED_COUNTRIES//|/ }
    </RequireAll>

    Options +Indexes
    AllowOverride None
</Directory>
EOF
    echo "Apache GeoIP restriction written to $APACHE_CONF"
    sudo a2enconf honeypot_geoip
    sudo systemctl reload apache2
    echo "Apache configuration reloaded."
fi

# -------------------- 5. Final note --------------------
echo ""
echo "Geographic gating is now active."
if [[ "$ALLOWED_COUNTRIES" == ".*" ]]; then
    echo "All countries are allowed (no restriction)."
else
    echo "Allowed country codes: ${*:1}"
fi
echo "Requests from other countries will receive HTTP 403 Forbidden."