#!/usr/bin/env bash
#
# generate_keys.sh – create the GPG key pairs used by the honeypot
# ---------------------------------------------------------------
# • weekly.key   – RSA‑2048 (easy, low‑cost target)
# • monthly.key  – RSA‑4096 (harder, high‑value vault)  OR Ed25519
# • public keys are exported for the web folder
#
# Run this script on your Linux workstation.
# It will prompt you for two passphrases – one simple, one strong.
# ----------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# ---------- 1. Helper to create a key ----------
create_key () {
    local KEY_TYPE=$1      # rsa2048, rsa4096, or ed25519
    local USER_ID=$2       # descriptive name/email
    local PASSPHRASE=$3    # passphrase (read from stdin)

    # Build a temporary batch file for --batch --gen-key
    cat > /tmp/gpg_batch <<EOF
%echo Generating ${KEY_TYPE^^} key for ${USER_ID}
Key-Type: ${KEY_TYPE}
Key-Length: ${KEY_TYPE#rsa}
Subkey-Type: ${KEY_TYPE}
Subkey-Length: ${KEY_TYPE#rsa}
Name-Real: ${USER_ID}
Expire-Date: 0
Passphrase: ${PASSPHRASE}
%commit
%echo done
EOF

    # Run GPG in batch mode (no interactive prompts)
    gpg --batch --gen-key /tmp/gpg_batch
    rm -f /tmp/gpg_batch
}

# ---------- 2. Prompt for passphrases ----------
echo "=== Weekly (RSA‑2048) key – keep this passphrase modest ==="
read -rsp "Enter passphrase (will not echo): " WEEKLY_PW
echo
echo "=== Monthly (RSA‑4096) key – use a strong, random phrase ==="
read -rsp "Enter passphrase (will not echo): " MONTHLY_PW
echo

# ---------- 3. Create the keys ----------
# Weekly key – RSA‑2048
create_key "rsa2048" "Honeypot Weekly 2026 <weekly@${HOSTNAME}>" "$WEEKLY_PW"

# Monthly key – RSA‑4096 (you can switch to ed25519 by changing the type)
create_key "rsa4096" "Honeypot Monthly 2026 <monthly@${HOSTNAME}>" "$MONTHLY_PW"

# ---------- 4. Export public keys ----------
OUT_DIR="$(dirname "$(realpath "$0")")/../web/honeypot/keys"
mkdir -p "$OUT_DIR"

gpg --armor --export "Honeypot Weekly 2026" > "$OUT_DIR/weekly_pub.asc"
gpg --armor --export "Honeypot Monthly 2026" > "$OUT_DIR/monthly_pub.asc"

# ---------- 5. Show fingerprints (useful for verification) ----------
echo "=== Public key fingerprints ==="
gpg --list-keys --with-colons "Honeypot Weekly 2026"   | awk -F: '/^fpr/ {print "weekly:", $10}'
gpg --list-keys --with-colons "Honeypot Monthly 2026"  | awk -F: '/^fpr/ {print "monthly:", $10}'

echo "Keys generated and public parts placed in:"
echo "  $OUT_DIR"
echo "You can now proceed to the next script (make_fragments.sh)."