#!/usr/bin/env bash
#
# create_self_destruct.sh – add expiration handling to the honeypot bundles
# ------------------------------------------------------------------------
# What it does
#   1. Scans the honeypot folder for *.asc files that don’t already have an
#      expiration tag in their name.
#   2. Renames each file to include a clear “Expires‑YYYYMMDD” suffix.
#   3. Writes a tiny cron entry that runs once a day, checks the dates,
#      moves expired files out of the public directory, and creates a
#      placeholder that triggers a 410 Gone response.
#   4. (Optional) Adds a minimal nginx or Apache location block that
#      returns 410 for any request that matches the placeholder file.
#
# Prerequisites
#   • You have already generated the conversation bundles (bundle_*.asc)
#   • Your web root is /var/www/html (adjust the variable if it differs)
#   • You have sudo rights on the machine where the web server runs
#
# Usage
#   ./create_self_destruct.sh   # runs once and installs the cron job
# ------------------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# -------------------- 1. Configuration --------------------
# Change these if your layout is different
WEB_ROOT="/var/www/html"
HONEYPOT_DIR="${WEB_ROOT}/honeypot"
CRON_FILE="/etc/cron.d/honeypot_selfdestruct"
NGINX_CONF="/etc/nginx/conf.d/honeypot_410.conf"
APACHE_CONF="/etc/apache2/conf-available/honeypot_410.conf"

# How many days a bundle should stay live after creation
LIVE_DAYS=30

# -------------------- 2. Add expiration suffixes --------------------
echo "Scanning ${HONEYPOT_DIR} for .asc files without an expiration tag…"

shopt -s nullglob   # avoid “no matches” errors
for FILE in "${HONEYPOT_DIR}"/*.asc; do
    BASENAME=$(basename "$FILE")
    # Skip files that already contain “Expires-”
    if [[ "$BASENAME" == *Expires-* ]]; then
        continue
    fi

    # Compute expiration date (today + LIVE_DAYS)
    EXPIRY=$(date -d "+${LIVE_DAYS} days" +"%Y%m%d")
    NEWNAME="${BASENAME%.asc}_Expires-${EXPIRY}.asc"

    mv "$FILE" "${HONEYPOT_DIR}/${NEWNAME}"
    echo " → Renamed $(basename "$FILE") → $NEWNAME (expires $EXPIRY)"
done
shopt -u nullglob

# -------------------- 3. Install the daily cron job --------------------
# The cron script will be tiny – we embed it directly in the crontab file
cat <<'EOF' > "$CRON_FILE"
# Honeypot self‑destruct – runs daily at 02:00 UTC
0 2 * * * root /usr/local/bin/honeypot_cleanup.sh
EOF

chmod 644 "$CRON_FILE"
echo "Cron file installed at $CRON_FILE"

# -------------------- 4. Write the cleanup helper script --------------------
CLEANUP_SCRIPT="/usr/local/bin/honeypot_cleanup.sh"
cat <<'EOS' > "$CLEANUP_SCRIPT"
#!/usr/bin/env bash
# honeypot_cleanup.sh – invoked by cron to purge expired bundles

WEB_ROOT="/var/www/html"
HONEYPOT_DIR="${WEB_ROOT}/honeypot"
TODAY=$(date +"%Y%m%d")

shopt -s nullglob
for FILE in "${HONEYPOT_DIR}"/*Expires-*.asc; do
    # Extract the YYYYMMDD part from the filename
    EXP=$(echo "$(basename "$FILE")" | grep -oE 'Expires-[0-9]{8}' | cut -d- -f2)
    if [[ "$EXP" < "$TODAY" ]]; then
        # Move the expired file to a hidden archive folder
        ARCHIVE_DIR="${HONEYPOT_DIR}/archived"
        mkdir -p "$ARCHIVE_DIR"
        mv "$FILE" "$ARCHIVE_DIR/"

        # Create a placeholder that will trigger a 410 response
        PLACEHOLDER="${HONEYPOT_DIR}/$(basename "$FILE")"
        touch "$PLACEHOLDER"
    fi
done
shopt -u nullglob
EOS

chmod +x "$CLEANUP_SCRIPT"
echo "Cleanup helper installed at $CLEANUP_SCRIPT"

# -------------------- 5. Configure 410 response (nginx) --------------------
if command -v nginx >/dev/null 2>&1; then
    cat <<'EOG' > "$NGINX_CONF"
# Return 410 Gone for any placeholder file created by the cleanup script
location ~* ^/honeypot/.*Expires-[0-9]{8}\.asc$ {
    if (-f $request_filename) {
        return 410 "File removed – expired.";
    }
}
EOG
    nginx -t && systemctl reload nginx
    echo "nginx 410 rule written to $NGINX_CONF and reloaded."
fi

# -------------------- 6. Configure 410 response (Apache) --------------------
if command -v apache2ctl >/dev/null 2>&1; then
    cat <<'EOG' > "$APACHE_CONF"
# Return 410 Gone for expired placeholders
<FilesMatch "Expires-[0-9]{8}\.asc$">
    Require all granted
    Header set Status "410 Gone"
    ErrorDocument 410 "File removed – expired."
</FilesMatch>
EOG
    a2enconf honeypot_410
    systemctl reload apache2
    echo "Apache 410 rule written to $APACHE_CONF and reloaded."
fi

# -------------------- 7. Summary --------------------
echo ""
echo "Self‑destruct setup complete."
echo "• Bundles now carry an Expires‑YYYYMMDD suffix."
echo "• A daily cron job will move expired files to /honeypot/archived."
echo "• Requests for an expired file will receive HTTP 410 Gone."
echo "You can adjust LIVE_DAYS at the top of this script if you need a"
echo "shorter or longer window."