#!/usr/bin/env bash
#
# bundle_conversations.sh – create the “encrypted‑conversation” honeypot files
# -------------------------------------------------------------------------
# What it does
#   1. Picks a random fragment from the pool you generated with make_fragments.sh.
#   2. Writes a short, realistic chat‑log (markdown style) that mentions the fragment.
#   3. Packages the log + fragment into a tarball.
#   4. Encrypts the tarball with the **weekly RSA‑2048** public key.
#   5. Places the final .asc file in the public web folder (web/honeypot/).
#
# Prerequisites
#   • generate_keys.sh has been run and the weekly key is in your GPG keyring.
#   • make_fragments.sh has created the fragment files in web/honeypot/fragments/.
#   • You are on Linux (the script uses standard GNU utilities that ship with it).
#
# Usage
#   ./bundle_conversations.sh   # creates one bundle
#   ./bundle_conversations.sh 5   # creates five bundles (optional count argument)
# -------------------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# ---------- 1. Where everything lives ----------
BASE_DIR="$(dirname "$(realpath "$0")")/.."            # project root
FRAG_DIR="$BASE_DIR/web/honeypot/fragments"
OUT_DIR="$BASE_DIR/web/honeypot"
TMP_DIR="$(mktemp -d)"                               # temporary workspace

# ---------- 2. How many bundles to make ----------
COUNT=${1:-1}   # default is 1 bundle if no argument is supplied

echo "Creating $COUNT encrypted‑conversation bundle(s)..."

for ((i=1; i<=COUNT; i++)); do
    # ----- 2a. Choose a random fragment -----
    FRAG_FILE=$(ls "$FRAG_DIR"/frag_*.bin | shuf -n1)
    FRAG_BASENAME=$(basename "$FRAG_FILE")

    # ----- 2b. Build a tiny chat log -----
    LOG_FILE="$TMP_DIR/chat_$(printf "%02d" "$i").md"
    cat > "$LOG_FILE" <<EOF
From: alice@example.com
To: bob@example.com
Subject: Quick check‑in
Date: $(date +"%Y-%m-%d %H:%M:%S %Z")

Hey Pix,

Just wanted to confirm you got the new key fragment. I’ve attached it as **$FRAG_BASENAME**. Let me know if the checksum matches.

Cheers,
Alice
EOF

    # ----- 2c. Assemble the bundle -----
    BUNDLE_NAME="bundle_$(printf "%02d" "$i").tar.gz"
    tar czf "$TMP_DIR/$BUNDLE_NAME" -C "$TMP_DIR" "$(basename "$LOG_FILE")" "$FRAG_BASENAME"

    # ----- 2d. Encrypt with the weekly RSA‑2048 key -----
    # The weekly key’s user‑id is “Honeypot Weekly 2026”
    ENC_NAME="bundle_$(printf "%02d" "$i").asc"
    gpg --armor --encrypt \
        --recipient "Honeypot Weekly 2026" \
        --output "$OUT_DIR/$ENC_NAME" \
        "$TMP_DIR/$BUNDLE_NAME"

    echo " → Created $ENC_NAME (contains $(basename "$LOG_FILE") + $FRAG_BASENAME)"
done

# ---------- 3. Clean up ----------
rm -rf "$TMP_DIR"

echo ""
echo "All bundles are now in $OUT_DIR"
echo "You can upload the .asc files to your web server (they’re already there if you ran this locally)."