#!/usr/bin/env bash
#
# build_snowflake.sh – assemble the non‑linear “snowflake” puzzle network
# -----------------------------------------------------------------------
# The network consists of four types of rooms:
#   • Entry rooms   – simple RSA‑2048 encrypted bundles (one fragment each)
#   • Bridge rooms  – RSA‑2048 bundles that require two different fragments
#   • Core room     – the layered Diffie‑Hellman trap (Room 3A)
#   • Vault room    – RSA‑4096 (or Ed25519) bundle that needs three fragments
#
# Prerequisites
#   • generate_keys.sh has created the weekly (RSA‑2048) and monthly (RSA‑4096) keys.
#   • make_fragments.sh has produced a pool of random fragments in web/honeypot/fragments/.
#   • bundle_conversations.sh has already created a handful of entry bundles.
#
# What the script does
#   1. Picks fragments from the pool and assigns them to rooms.
#   2. Writes tiny markdown “room notes” that describe the dependency.
#   3. Packages each room (note + required fragments) into a tarball.
#   4. Encrypts the tarball with the appropriate key (RSA‑2048 for entry/bridge,
#      RSA‑4096 for the vault, and the DH trap stays unencrypted but its
#      parameters are inside a PGP‑encrypted file).
#   5. Drops the final .asc files into web/honeypot/ so they appear alongside the
#      regular conversation bundles.
#
# Run it once after you have generated the entry bundles.
# ------------------------------------------------------------------------

set -euo pipefail   # stop on errors, undefined vars, or pipe failures

# -------------------- 1. Paths and helpers --------------------
BASE_DIR="$(dirname "$(realpath "$0")")/.."          # project root
FRAG_DIR="${BASE_DIR}/web/honeypot/fragments"
OUT_DIR="${BASE_DIR}/web/honeypot"
TMP_DIR="$(mktemp -d)"                               # temporary workspace

# Helper: pick a random fragment and return its filename
pick_fragment () {
    ls "${FRAG_DIR}"/frag_*.bin | shuf -n1
}

# Helper: create a markdown note describing the room
write_note () {
    local ROOM_NAME=$1
    local DEP_DESC=$2
    local NOTE_PATH=$3

    cat > "$NOTE_PATH" <<EOF
# ${ROOM_NAME}

This room is part of the snowflake puzzle. ${DEP_DESC}

To unlock it you must provide the required fragment(s) listed in the
accompanying files. Once decrypted you will find the next fragment
that can be used in another room.

Good luck.
EOF
}

# -------------------- 2. ENTRY ROOMS (simple) --------------------
echo "Creating entry rooms (RSA‑2048)…"
for i in {1..4}; do
    FRAG=$(pick_fragment)
    NOTE="${TMP_DIR}/entry_${i}.md"
    write_note "Entry Room ${i}" \
        "It contains a single OTP fragment (**${FRAG##*/}**) and no other dependencies." \
        "$NOTE"

    BUNDLE="${TMP_DIR}/entry_${i}.tar.gz"
    tar czf "$BUNDLE" -C "$TMP_DIR" "$(basename "$NOTE")" "$(basename "$FRAG")"

    gpg --armor --encrypt \
        --recipient "Honeypot Weekly 2026" \
        --output "${OUT_DIR}/entry_${i}.asc" "$BUNDLE"
    echo " → entry_${i}.asc created"
done

# -------------------- 3. BRIDGE ROOMS (require two fragments) --------------------
echo "Creating bridge rooms (RSA‑2048, two‑fragment requirement)…"
for i in {1..2}; do
    FRAG_A=$(pick_fragment)
    FRAG_B=$(pick_fragment)
    # Ensure we don’t pick the same file twice
    while [[ "$FRAG_A" == "$FRAG_B" ]]; do
        FRAG_B=$(pick_fragment)
    done

    NOTE="${TMP_DIR}/bridge_${i}.md"
    write_note "Bridge Room ${i}" \
        "It requires **${FRAG_A##*/}** and **${FRAG_B##*/}** to be concatenated (or XOR‑ed) before decryption." \
        "$NOTE"

    BUNDLE="${TMP_DIR}/bridge_${i}.tar.gz"
    tar czf "$BUNDLE" -C "$TMP_DIR" "$(basename "$NOTE")" "$(basename "$FRAG_A")" "$(basename "$FRAG_B")"

    gpg --armor --encrypt \
        --recipient "Honeypot Weekly 2026" \
        --output "${OUT_DIR}/bridge_${i}.asc" "$BUNDLE"
    echo " → bridge_${i}.asc created"
done

# -------------------- 4. CORE ROOM – Layered DH trap (Room 3A) --------------------
echo "Creating core room (Layered Diffie‑Hellman trap)…"

# Generate a malformed DH parameter set (small script)
DH_PARAMS="${TMP_DIR}/dh_params.txt"
cat > "$DH_PARAMS" <<'EOF'
# Intentionally malformed Diffie‑Hellman parameters
# p (prime) is not actually prime, g (generator) is out of range
p=FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
g=123456789ABCDEF
EOF

CORE_NOTE="${TMP_DIR}/core_room.md"
write_note "Core Room (DH Trap)" \
    "Contains malformed Diffie‑Hellman parameters that force a fallback calculation. The resulting fragment (**core_frag.bin**) is produced after the trap is solved." \
    "$CORE_NOTE"

# Create a dummy fragment that will be revealed after the trap
touch "${TMP_DIR}/core_frag.bin"
dd if=/dev/urandom bs=24 count=1 of="${TMP_DIR}/core_frag.bin" status=none

CORE_BUNDLE="${TMP_DIR}/core_room.tar.gz"
tar czf "$CORE_BUNDLE" -C "$TMP_DIR" "$(basename "$CORE_NOTE")" "$(basename "$DH_PARAMS")" core_frag.bin

# Encrypt the core room with the weekly RSA‑2048 key (still “easy” to get into)
gpg --armor --encrypt \
    --recipient "Honeypot Weekly 2026" \
    --output "${OUT_DIR}/core_room.asc" "$CORE_BUNDLE"
echo " → core_room.asc created"

# -------------------- 5. VAULT ROOM – high‑value (RSA‑4096) --------------------
echo "Creating vault room (RSA‑4096, three‑fragment requirement)…"

# Pick three distinct fragments (could be from entry or bridge rooms)
VAULT_FRAG1=$(pick_fragment)
VAULT_FRAG2=$(pick_fragment)
while [[ "$VAULT_FRAG2" == "$VAULT_FRAG1" ]]; do
    VAULT_FRAG2=$(pick_fragment)
done
VAULT_FRAG3=$(pick_fragment)
while [[ "$VAULT_FRAG3" == "$VAULT_FRAG1" || "$VAULT_FRAG3" == "$VAULT_FRAG2" ]]; do
    VAULT_FRAG3=$(pick_fragment)
done

VAULT_NOTE="${TMP_DIR}/vault_room.md"
write_note "Vault Room (High‑Value)" \
    "To open this vault you need **${VAULT_FRAG1##*/}**, **${VAULT_FRAG2##*/}**, and **${VAULT_FRAG3##*/}**. Once decrypted you will find a “dead‑end” token (a random hash) that serves as the final prize." \
    "$VAULT_NOTE"

VAULT_BUNDLE="${TMP_DIR}/vault_room.tar.gz"
tar czf "$VAULT_BUNDLE" -C "$TMP_DIR" "$(basename "$VAULT_NOTE")" "$(basename "$VAULT_FRAG1")" "$(basename "$VAULT_FRAG2")" "$(basename "$VAULT_FRAG3")"

# Encrypt with the stronger monthly key
gpg --armor --encrypt \
    --recipient "Honeypot Monthly 2026" \
    --output "${OUT_DIR}/vault_room.asc" "$VAULT_BUNDLE"
echo " → vault_room.asc created"

# -------------------- 6. Cleanup --------------------
rm -rf "$TMP_DIR"
echo ""
echo "Snowflake puzzle assembled! All .asc files are now in ${OUT_DIR}"
echo "You can upload the folder to your web server (they’re already there if you ran locally)."