# tests/test_persona.py
"""
Unit tests for the ``Persona`` class.

Run with:
    pytest -vv tests/test_persona.py
"""

import json
import re
import time
from datetime import datetime

import pytest

from src.persona import Persona


# ----------------------------------------------------------------------
# Helper – simple regex to validate ISO‑8601 timestamps (UTC offset optional)
# ----------------------------------------------------------------------
ISO8601_REGEX = re.compile(
    r"^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}"
    r"(?:\.\d+)?(?:Z|[+\-]\d{2}:\d{2})?$"
)


# ----------------------------------------------------------------------
@pytest.fixture
def minimal_cfg(tmp_path):
    """
    Provide a tiny but complete persona configuration dictionary.
    The fixture writes a temporary ``api_keys.env`` file so that any
    accidental API calls (which shouldn’t happen in these tests) will
    raise a clear error.
    """
    cfg = {
        "user_agents": {
            "phone":   "Mozilla/5.0 (iPhone; CPU iPhone OS 18_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148",
            "laptop":  "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/134.0.0.0 Safari/537.36",
            "watch":   "Mozilla/5.0 (Linux; Android 12; Wear OS) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/134.0.0.0 Mobile Safari/537.36"
        },
        "language": "en-US",
        "screen_resolution": "1920x1080",
        "color_depth": 24,
        "plugins": ["Chrome PDF Viewer", "Widevine Content Decryption Module"],
        "canvas_hash": "a3f5c9e1d2b4f6a7c8e9d0b1c2d3e4f5a6b7c8d9e0f1a2b3c4d5e6f7a8b9c0d1",
        "webgl_renderer": "ANGLE (Intel(R) UHD Graphics 620 Direct3D11 vs_5_0 ps_5_0)",
        "referrer_policy": "strict-origin-when-cross-origin",
        "cache_control": {"max_age": 0, "no_cache": True},
        "navigator": {"hardwareConcurrency": 8, "deviceMemory": 8},
        "baseline_heart_rate": 72,
        "baseline_steps_per_minute": 100,
        "stride_meters": 0.78,
        "battery_capacity": {"phone": 100, "laptop": 100, "watch": 100}
    }
    return cfg


# ----------------------------------------------------------------------
def test_initialisation(minimal_cfg):
    """Persona should load all fingerprint fields and set defaults."""
    p = Persona(minimal_cfg)

    # Fingerprint sanity checks
    assert p.ua["phone"].startswith("Mozilla/5.0 (iPhone")
    assert p.language == "en-US"
    assert p.screen_resolution == "1920x1080"
    assert p.color_depth == 24
    assert "Chrome PDF Viewer" in p.plugins
    assert p.canvas_hash.startswith("a3f5c9e1")
    assert p.webgl_renderer.startswith("ANGLE")
    assert p.referrer_policy == "strict-origin-when-cross-origin"
    assert p.cache_control["no_cache"] is True

    # Biometric defaults
    assert p.baseline_hr == 72
    assert p.current_hr["watch"] == 72
    assert p.step_counter == 0

    # Battery defaults
    assert p.battery["phone"] == 100
    assert p.battery["laptop"] == 100
    assert p.battery["watch"] == 100

    # Location defaults (should be neutral)
    assert p.lat == 0.0 and p.lon == 0.0
    assert p.tz == "UTC"


# ----------------------------------------------------------------------
def test_location_and_timestamp(minimal_cfg):
    """Setting location updates timezone and timestamp format."""
    p = Persona(minimal_cfg)
    # London coordinates & timezone
    p.set_location(51.4700, -0.4543, "Europe/London")
    assert p.lat == 51.4700
    assert p.lon == -0.4543
    assert p.tz == "Europe/London"

    ts = p.get_timestamp()
    # Verify ISO‑8601 format (e.g., 2025-12-21T14:32:07+00:00)
    assert ISO8601_REGEX.match(ts), f"Timestamp not ISO‑8601: {ts}"


# ----------------------------------------------------------------------
def test_step_and_heart_rate_logic(minimal_cfg):
    """Adding steps should increase the counter and optionally raise HR."""
    p = Persona(minimal_cfg)

    # Initial state
    assert p.step_counter == 0
    init_hr = p.current_hr["watch"]

    # Add 120 steps (≈ 2 minutes of walking at 60 spm)
    p.add_steps(120)
    assert p.step_counter == 120

    # Simulate a modest HR bump (5 bpm per 30 steps → +20 bpm)
    new_hr = p.update_heart_rate("watch", +20)
    assert new_hr == init_hr + 20
    # Ensure the HR stays within realistic bounds (50‑190)
    p.update_heart_rate("watch", -200)   # huge negative delta
    assert p.current_hr["watch"] >= 50


# ----------------------------------------------------------------------
def test_battery_drain_and_charge(minimal_cfg):
    """Battery should never exceed 100 % or drop below 0 %."""
    p = Persona(minimal_cfg)

    # Drain 30 % from the phone
    p.drain_battery("phone", 30)
    assert p.battery["phone"] == 70

    # Over‑drain – should clamp at 0
    p.drain_battery("phone", 100)
    assert p.battery["phone"] == 0

    # Charge back up – clamp at 100
    p.charge_battery("phone", 150)
    assert p.battery["phone"] == 100


# ----------------------------------------------------------------------
def test_clipboard_sync_and_retrieval(minimal_cfg):
    """Copying to clipboard should store the latest entry and be retrievable."""
    p = Persona(minimal_cfg)

    # Initially empty
    assert p.get_clipboard() is None

    entry = p.copy_to_clipboard(
        source_device="phone",
        content_type="url",
        content="https://example.com/flight/AA123",
        description="Flight status page"
    )
    # Verify the stored structure
    assert entry["source_device"] == "phone"
    assert entry["content_type"] == "url"
    assert entry["content"] == "https://example.com/flight/AA123"

    # Retrieval should give the same dict (reference)
    retrieved = p.get_clipboard()
    assert retrieved is entry
    assert retrieved["description"] == "Flight status page"


# ----------------------------------------------------------------------
def test_serialisation_of_persona_state(minimal_cfg):
    """Ensure the Persona can be JSON‑serialised (useful for logs)."""
    p = Persona(minimal_cfg)
    p.set_location(40.6413, -73.7781, "America/New_York")
    p.add_steps(200)
    p.update_heart_rate("watch", +5)

    # Build a shallow dict representation
    state = {
        "id": p.id,
        "location": {"lat": p.lat, "lon": p.lon, "tz": p.tz},
        "steps": p.step_counter,
        "heart_rate": p.current_hr,
        "battery": p.battery,
        "clipboard": p.get_clipboard()
    }

    # Serialise to JSON – should not raise
    json_str = json.dumps(state)
    assert isinstance(json_str, str)

    # Round‑trip check
    loaded = json.loads(json_str)
    assert loaded["location"]["tz"] == "America/New_York"
    assert loaded["steps"] == 200