# tests/test_itinerary.py
"""
Tests for the ``ItineraryBuilder`` class.

The tests focus on the *logic* of the itinerary planner:
* season detection (via ``wettr``)
* filtering destinations by season
* budget‑weighting logic
* selection of a real‑time flight (mocked ``aviationstack`` call)
* updating the shared ``Persona`` with the new geographic state
* proper error handling when no flight can be found

External network calls are **patched** with ``monkeypatch`` so the test
suite runs offline and deterministically.
"""

import datetime as dt
import json
import re
import pytest

from src.persona import Persona
from src.itinerary import ItineraryBuilder

# ----------------------------------------------------------------------
# Fixtures – minimal configuration data used by the tests
# ----------------------------------------------------------------------
@pytest.fixture
def minimal_persona_cfg():
    """A tiny but complete persona configuration (same as in test_persona)."""
    return {
        "user_agents": {
            "phone":   "Mozilla/5.0 (iPhone; CPU iPhone OS 18_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148",
            "laptop":  "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/134.0.0.0 Safari/537.36",
            "watch":   "Mozilla/5.0 (Linux; Android 12; Wear OS) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/134.0.0.0 Mobile Safari/537.36"
        },
        "language": "en-US",
        "screen_resolution": "1920x1080",
        "color_depth": 24,
        "plugins": ["Chrome PDF Viewer", "Widevine Content Decryption Module"],
        "canvas_hash": "a3f5c9e1d2b4f6a7c8e9d0b1c2d3e4f5a6b7c8d9e0f1a2b3c4d5e6f7a8b9c0d1",
        "webgl_renderer": "ANGLE (Intel(R) UHD Graphics 620 Direct3D11 vs_5_0 ps_5_0)",
        "referrer_policy": "strict-origin-when-cross-origin",
        "cache_control": {"max_age": 0, "no_cache": True},
        "navigator": {"hardwareConcurrency": 8, "deviceMemory": 8},
        "baseline_heart_rate": 72,
        "baseline_steps_per_minute": 100,
        "stride_meters": 0.78,
        "battery_capacity": {"phone": 100, "laptop": 100, "watch": 100}
    }


@pytest.fixture
def destinations_cfg():
    """
    A small pool of destinations that includes all three seasons.
    The ``season`` field drives the filter in ``ItineraryBuilder``.
    """
    return [
        {"city": "London", "iata": "LHR", "season": "any",   "budget": "mid"},
        {"city": "Nassau", "iata": "NAS", "season": "winter","budget": "low"},
        {"city": "Tokyo",  "iata": "HND", "season": "any",   "budget": "high"},
        {"city": "Portland, Maine", "iata": "PWM", "season": "summer","budget": "low"},
    ]


@pytest.fixture
def airports_cfg():
    """
    Minimal airport metadata – only the fields accessed by the itinerary
    builder are required.
    """
    return {
        "LHR": {
            "airport_id": "LHR",
            "city": "London",
            "country": "GB",
            "lat": 51.4700,
            "lon": -0.4543,
            "tz": "Europe/London",
            "wifi_ssids": ["Free_WiFi_LHR", "British_Airways_WiFi"]
        },
        "NAS": {
            "airport_id": "NAS",
            "city": "Nassau",
            "country": "BS",
            "lat": 25.0380,
            "lon": -77.4667,
            "tz": "America/Nassau",
            "wifi_ssids": ["Free_WiFi_NAS"]
        },
        "HND": {
            "airport_id": "HND",
            "city": "Tokyo",
            "country": "JP",
            "lat": 35.5494,
            "lon": 139.7798,
            "tz": "Asia/Tokyo",
            "wifi_ssids": ["Free_WiFi_HND"]
        },
        "PWM": {
            "airport_id": "PWM",
            "city": "Portland, Maine",
            "country": "US",
            "lat": 43.6462,
            "lon": -70.3090,
            "tz": "America/New_York",
            "wifi_ssids": ["Free_WiFi_PWM"]
        }
    }


# ----------------------------------------------------------------------
# Helper – mock ``wettr`` temperature function
# ----------------------------------------------------------------------
def mock_wettr_temp_cold(city):
    """Return a cold temperature (≤ 10 °C) to force a “winter” season."""
    return 5   # Celsius


def mock_wettr_temp_hot(city):
    """Return a hot temperature (≥ 30 °C) to force a “summer” season."""
    return 32  # Celsius


# ----------------------------------------------------------------------
# Helper – mock ``aviationstack.get_flight_info``
# ----------------------------------------------------------------------
def mock_flight_info_success(dep_iata, arr_iata, date):
    """Return a deterministic flight dict for any input."""
    return {
        "flight_number": f"{dep_iata}{arr_iata}123",
        "airline": {"name": "MockAir"},
        "airline_iata": "MA",
        "operating_airline_iata": "MA",
        "aircraft": {"model": "B777"},
        "departure": {
            "scheduled": f"{date}T08:00:00",
            "actual": None,
            "delay": 0,
            "gate": "A12"
        },
        "arrival": {
            "scheduled": f"{date}T12:00:00",
            "actual": None,
            "delay": 0,
            "gate": "B34"
        },
        "flight_status": "scheduled",
        "date": date,
        "status_url": f"https://mockair.com/status/{dep_iata}{arr_iata}123"
    }


def mock_flight_info_none(dep_iata, arr_iata, date):
    """Simulate no matching flight (e.g., API returns empty list)."""
    return None


# ----------------------------------------------------------------------
# Test 1 – season detection (cold → winter)
# ----------------------------------------------------------------------
def test_season_filter_winter(monkeypatch, minimal_persona_cfg,
                              destinations_cfg, airports_cfg):
    # Patch wettr to return a cold temperature → winter
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", mock_wettr_temp_cold)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(destinations_cfg, airports_cfg, persona)

    # Force the home location to be a city that will be evaluated as winter
    # (the actual city name is irrelevant because we patched wettr)
    builder.persona.set_location(0.0, 0.0, "UTC")  # dummy location

    # Run the planning step – it should filter out the “summer‑only” entry
    builder.plan_next_leg()
    # The chosen destination must be either LHR, NAS or HND (all winter‑eligible)
    assert builder.destination["iata"] in {"LHR", "NAS", "HND"}

    # Verify that the persona's location has been updated to the departure airport
    dep_iata = builder.current_airport_iata
    assert dep_iata in airports_cfg
    dep_info = airports_cfg[dep_iata]
    assert persona.lat == dep_info["lat"]
    assert persona.lon == dep_info["lon"]
    assert persona.tz == dep_info["tz"]


# ----------------------------------------------------------------------
# Test 2 – season detection (hot → summer)
# ----------------------------------------------------------------------
def test_season_filter_summer(monkeypatch, minimal_persona_cfg,
                              destinations_cfg, airports_cfg):
    # Patch wettr to return a hot temperature → summer
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", mock_wettr_temp_hot)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(destinations_cfg, airports_cfg, persona)

    # Run the planning step – now the “summer‑only” entry (PWM) should be
    # eligible, while the winter‑only entry (NAS) should be excluded.
    builder.plan_next_leg()
    assert builder.destination["iata"] in {"LHR", "PWM", "HND"}

    # Ensure the chosen airport metadata exists
    dep_iata = builder.current_airport_iata
    assert dep_iata in airports_cfg


# ----------------------------------------------------------------------
# Test 3 – budget weighting (low > mid > high)
# ----------------------------------------------------------------------
def test_budget_weighting(monkeypatch, minimal_persona_cfg,
                         destinations_cfg, airports_cfg):
    # Use a neutral season (wettr returns 20 °C → “shoulder” → any season)
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", lambda _: 20)

    # Override destinations to have distinct budgets only
    test_dest = [
        {"city": "CheapTown", "iata": "CT1", "season": "any", "budget": "low"},
        {"city": "MidVille",   "iata": "MV1", "season": "any", "budget": "mid"},
        {"city": "ExpensiveCity", "iata": "EC1", "season": "any", "budget": "high"},
    ]
    # Add dummy airport entries for the three IATA codes
    test_airports = {
        "CT1": {"airport_id": "CT1", "city": "CheapTown", "country": "ZZ",
                "lat": 0.0, "lon": 0.0, "tz": "UTC", "wifi_ssids": ["Free_WiFi_CT1"]},
        "MV1": {"airport_id": "MV1", "city": "MidVille", "country": "ZZ",
                "lat": 1.0, "lon": 1.0, "tz": "UTC", "wifi_ssids": ["Free_WiFi_MV1"]},
        "EC1": {"airport_id": "EC1", "city": "ExpensiveCity", "country": "ZZ",
                "lat": 2.0, "lon": 2.0, "tz": "UTC", "wifi_ssids": ["Free_WiFi_EC1"]},
    }

    # Mock flight info to always succeed
    from src.api_clients import aviationstack
    monkeypatch.setattr(aviationstack, "get_flight_info", mock_flight_info_success)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(test_dest, test_airports, persona)

    # Run the planner many times to observe the weighting distribution
    selections = {"CT1": 0, "MV1": 0, "EC1": 0}
    for _ in range(300):
        builder.plan_next_leg()
        selections[builder.destination["iata"]] += 1

    # Expect low‑budget (CT1) to be selected roughly 3× more often than mid,
    # and mid roughly 2× more often than high.
    low = selections["CT1"]
    mid = selections["MV1"]
    high = selections["EC1"]

    # Simple ratio checks (allowing some statistical variance)
    assert low > mid > high
    assert low / mid > 1.5   # low should be at least ~1.5× mid
    assert mid / high > 1.2  # mid should be at least ~1.2× high


# ----------------------------------------------------------------------
# Test 4 – flight lookup success path
# ----------------------------------------------------------------------
def test_flight_lookup_success(monkeypatch, minimal_persona_cfg,
                               destinations_cfg, airports_cfg):
    # Force a neutral season so any destination can be chosen
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", lambda _: 20)

    # Mock the AviationStack call to return a deterministic flight
    from src.api_clients import aviationstack
    monkeypatch.setattr(aviationstack, "get_flight_info", mock_flight_info_success)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(destinations_cfg, airports_cfg, persona)

    builder.plan_next_leg()

    # Verify that a flight dict was attached to the persona
    assert hasattr(persona, "next_flight")
    flight = persona.next_flight
    assert flight["flight_number"].endswith("123")
    assert flight["airline"] == "MockAir"
    assert flight["departure"]["gate"] is not None

    # Ensure the destination airport metadata is stored
    assert hasattr(persona, "destination_airport")
    dest_meta = persona.destination_airport
    assert dest_meta["airport_id"] == builder.destination["iata"]


# ----------------------------------------------------------------------
# Test 5 – flight lookup failure raises RuntimeError
# ----------------------------------------------------------------------
def test_flight_lookup_failure(monkeypatch, minimal_persona_cfg,
                               destinations_cfg, airports_cfg):
    # Neutral season
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", lambda _: 20)

    # Mock AviationStack to always return None (no flight)
    from src.api_clients import aviationstack
    monkeypatch.setattr(aviationstack, "get_flight_info", mock_flight_info_none)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(destinations_cfg, airports_cfg, persona)

    with pytest.raises(RuntimeError, match="Could not find a real flight"):
        builder.plan_next_leg()


# ----------------------------------------------------------------------
# Test 6 – itinerary builder returns a consistent info dict
# ----------------------------------------------------------------------
def test_get_current_leg_info(monkeypatch, minimal_persona_cfg,
                              destinations_cfg, airports_cfg):
    # Neutral season
    from src.api_clients import wettr
    monkeypatch.setattr(wettr, "get_temperature", lambda _: 20)

    # Mock flight info
    from src.api_clients import aviationstack
    monkeypatch.setattr(aviationstack, "get_flight_info", mock_flight_info_success)

    persona = Persona(minimal_persona_cfg)
    builder = ItineraryBuilder(destinations_cfg, airports_cfg, persona)

    builder.plan_next_leg()
    leg_info = builder.get_current_leg_info()

    # Validate the structure of the returned dict
    assert "departure_airport" in leg_info
    assert "destination_airport" in leg_info
    assert "flight" in leg_info
    assert "destination_meta" in leg_info

    # Types
    assert isinstance(leg_info["departure_airport"], str)
    assert isinstance(leg_info["destination_airport"], str)
    assert isinstance(leg_info["flight"], dict)
    assert isinstance(leg_info["destination_meta"], dict)

    # Consistency checks
    assert leg_info["flight"]["flight_number"] == persona.next_flight["flight_number"]
    assert leg_info["destination_airport"] == builder.destination["iata"]