# tests/test_guards.py
"""
Unit tests for the guard functions defined in ``src/guards.py``.

The tests cover:

* Wi‑Fi ↔ venue consistency (`guard_wifi_location`)
* Step‑to‑distance sanity (`guard_step_distance`)
* Heart‑rate ↔ activity mapping (`guard_hr_activity`)
* Battery‑drain / charge logic (`guard_battery`)
* Monotonic timestamp enforcement (`guard_timestamp_monotonic`)
* Clipboard size / type validation (`guard_clipboard`)
* Cross‑device timestamp cohesion (`guard_cross_device_sync`)
* The high‑level dispatcher ``run_all_guards`` (basic integration)

All tests are pure‑Python – no external network calls.
"""

import copy
import datetime
import json
import pytest

from src import guards


# ----------------------------------------------------------------------
# Helper – a minimal mock Persona object with the attributes the guards
# expect.  Only the fields used in the tests are implemented.
# ----------------------------------------------------------------------
class MockPersona:
    def __init__(self):
        self.last_timestamps = {}          # device → last timestamp string
        self.current_venue = None          # dict with "wifi_ssids"
        self.prev_location_state = {"lat": 0.0, "lon": 0.0, "steps": 0}
        self.current_activity = "seated"
        self.battery = {"phone": 100, "laptop": 100, "watch": 100}
        self.baseline_hr = 72
        self.current_hr = {"phone": 72, "laptop": 72, "watch": 72}


# ----------------------------------------------------------------------
# 1️⃣ guard_wifi_location
# ----------------------------------------------------------------------
def test_guard_wifi_location_corrects_wrong_ssid():
    device_state = {"device": "phone", "ssid": "UNKNOWN_SSID", "rssi": -20}
    venue = {
        "wifi_ssids": ["Free_WiFi_LHR", "British_Airways_WiFi"],
        "name": "Heathrow"
    }

    corrected = guards.guard_wifi_location(device_state, venue)
    assert corrected["ssid"] in venue["wifi_ssids"]
    # RSSI should be clamped to the indoor range (-30 .. -90)
    assert -90 <= corrected["rssi"] <= -30


def test_guard_wifi_location_no_change_when_valid():
    device_state = {"device": "laptop", "ssid": "Free_WiFi_LHR", "rssi": -55}
    venue = {"wifi_ssids": ["Free_WiFi_LHR", "Delta_SkyClub_WiFi"]}

    corrected = guards.guard_wifi_location(device_state, venue)
    assert corrected["ssid"] == "Free_WiFi_LHR"
    assert corrected["rssi"] == -55


# ----------------------------------------------------------------------
# 2️⃣ guard_step_distance
# ----------------------------------------------------------------------
def test_guard_step_distance_corrects_mismatch():
    prev = {"lat": 51.4700, "lon": -0.4543, "steps": 0}
    # Move ~200 m north (≈ 0.0018° latitude)
    curr = {"lat": 51.4718, "lon": -0.4543, "steps": 10}
    corrected = guards.guard_step_distance(prev, curr, avg_stride_m=0.78)

    # Expected steps ≈ distance / stride ≈ 200 m / 0.78 ≈ 256 steps
    expected_steps = int(200 / 0.78)
    assert corrected["steps"] == prev["steps"] + expected_steps


def test_guard_step_distance_within_tolerance_no_change():
    prev = {"lat": 40.6413, "lon": -73.7781, "steps": 100}
    # Small move (~5 m) that should be within 15 % tolerance
    curr = {"lat": 40.64135, "lon": -73.7781, "steps": 105}
    corrected = guards.guard_step_distance(prev, curr, avg_stride_m=0.78)

    # Steps should stay as originally reported (105)
    assert corrected["steps"] == 105


# ----------------------------------------------------------------------
# 3️⃣ guard_hr_activity
# ----------------------------------------------------------------------
@pytest.mark.parametrize(
    "activity,initial_hr,expected_range",
    [
        ("walking", 72, (77, 92)),          # +5 … +20
        ("running", 72, (92, 112)),         # +20 … +40
        ("seated", 72, (67, 77)),           # -5 … +5
        ("security_assist", 72, (77, 84)),  # +5 … +12
        ("seatbelt_alert", 72, (77, 84)),   # same as above
        ("elevator", 72, (70, 74)),         # -2 … +2
        ("conveyor_belt", 72, (70, 74)),
    ]
)
def test_guard_hr_activity_bounds(activity, initial_hr, expected_range):
    device_state = {"heart_rate": initial_hr}
    corrected = guards.guard_hr_activity(device_state, activity)
    low, high = expected_range
    assert low <= corrected["heart_rate"] <= high


def test_guard_hr_activity_unknown_activity_defaults():
    device_state = {"heart_rate": 80}
    corrected = guards.guard_hr_activity(device_state, "unknown_activity")
    # Default range is baseline ±5 (baseline is 70 in our mock)
    assert 65 <= corrected["heart_rate"] <= 75


# ----------------------------------------------------------------------
# 4️⃣ guard_battery
# ----------------------------------------------------------------------
def test_guard_battery_charging_and_regular_use():
    device_state = {"battery_percent": 50, "charging": False}
    # Simulate regular use (drain)
    after_use = guards.guard_battery(device_state, "regular_use")
    assert after_use["battery_percent"] == 49.9  # 0.1 % drain

    # Start charging
    after_start = guards.guard_battery(after_use, "charging_start")
    assert after_start["charging"] is True

    # Stop charging
    after_stop = guards.guard_battery(after_start, "charging_stop")
    assert after_stop["charging"] is False


def test_guard_battery_never_exceeds_bounds():
    device_state = {"battery_percent": 99.9}
    # Simulate a tiny charge that would push it over 100
    after = guards.guard_battery(device_state, "regular_use")
    # Regular use only drains, so still ≤ 99.9
    assert after["battery_percent"] <= 100

    # Force a manual over‑charge
    device_state["battery_percent"] = 101
    after = guards.guard_battery(device_state, "regular_use")
    assert after["battery_percent"] == 100  # clamped


# ----------------------------------------------------------------------
# 5️⃣ guard_timestamp_monotonic
# ----------------------------------------------------------------------
def test_guard_timestamp_monotonic_corrects_backwards():
    prev = "2025-12-21T10:00:00+00:00"
    new = "2025-12-21T09:59:59+00:00"  # earlier than prev
    corrected = guards.guard_timestamp_monotonic(prev, new, "phone")
    # The corrected timestamp should be later than prev
    assert datetime.datetime.fromisoformat(corrected) > datetime.datetime.fromisoformat(prev)


def test_guard_timestamp_monotonic_no_change_when_forward():
    prev = "2025-12-21T10:00:00+00:00"
    new = "2025-12-21T10:00:05+00:00"
    corrected = guards.guard_timestamp_monotonic(prev, new, "laptop")
    assert corrected == new


# ----------------------------------------------------------------------
# 6️⃣ guard_clipboard
# ----------------------------------------------------------------------
def test_guard_clipboard_truncates_oversized_content():
    long_text = "A" * 12_000  # 12 KB > 10 KB limit
    entry = {"content_type": "text", "content": long_text, "description": "big blob"}
    corrected = guards.guard_clipboard(entry)

    # Content should be truncated and end with the marker
    assert corrected["content"].endswith("...[truncated]")
    # Size after truncation should be ≤ 10 KB
    assert len(corrected["content"].encode("utf-8")) <= 10 * 1024


def test_guard_clipboard_invalid_type_is_fixed():
    entry = {"content_type": "binary_blob", "content": "data"}
    corrected = guards.guard_clipboard(entry)
    assert corrected["content_type"] == "text"


def test_guard_clipboard_non_string_content_is_converted():
    entry = {"content_type": "url", "content": 12345}
    corrected = guards.guard_clipboard(entry)
    assert isinstance(corrected["content"], str)
    assert corrected["content"] == "12345"


# ----------------------------------------------------------------------
# 7️⃣ guard_cross_device_sync
# ----------------------------------------------------------------------
def test_guard_cross_device_sync_adjusts_outliers():
    base_ts = datetime.datetime(2025, 12, 21, 12, 0, 0)
    # Device A is on time, Device B is 5 seconds late (beyond 2‑second window)
    events = [
        {"device": "phone", "timestamp": base_ts.isoformat()},
        {"device": "laptop", "timestamp": (base_ts + datetime.timedelta(seconds=5)).isoformat()},
        {"device": "watch", "timestamp": base_ts.isoformat()},
    ]
    corrected = guards.guard_cross_device_sync(events, max_delta_seconds=2)

    # After correction, all timestamps should equal the median (base_ts)
    for ev in corrected:
        assert ev["timestamp"] == base_ts.isoformat()


def test_guard_cross_device_sync_leaves_good_events_untouched():
    base_ts = datetime.datetime(2025, 12, 21, 12, 0, 0)
    events = [
        {"device": "phone", "timestamp": base_ts.isoformat()},
        {"device": "laptop", "timestamp": (base_ts + datetime.timedelta(seconds=1)).isoformat()},
        {"device": "watch", "timestamp": (base_ts + datetime.timedelta(seconds=2)).isoformat()},
    ]
    corrected = guards.guard_cross_device_sync(events, max_delta_seconds=3)
    # No changes expected
    assert corrected == events


# ----------------------------------------------------------------------
# 8️⃣ run_all_guards – integration test (basic flow)
# ----------------------------------------------------------------------
def test_run_all_guards_sequence(monkeypatch):
    """
    Verify that ``run_all_guards`` invokes the appropriate guard functions
    and returns a payload that contains the modifications.
    """
    persona = MockPersona()
    persona.current_venue = {"wifi_ssids": ["Free_WiFi_Test"], "name": "Test Airport"}
    persona.prev_location_state = {"lat": 0.0, "lon": 0.0, "steps": 0}
    persona.current_activity = "walking"

    # Input payload mimics a Wi‑Fi connect event
    payload = {
        "device": "phone",
        "event": "wifi_connect",
        "timestamp": "2025-12-21T08:00:00+00:00",
        "ssid": "WRONG_SSID",
        "bssid": "02:42:00:00:00:01",
        "rssi": -20,
    }

    # Monkey‑patch the individual guards to spy on calls
    called = {"wifi": False, "step": False, "hr": False}
    def fake_wifi(dev_state, venue):
        called["wifi"] = True
        return dev_state
    def fake_step(prev, curr):
        called["step"] = True
        return curr
    def fake_hr(dev_state, activity):
        called["hr"] = True
        return dev_state

    monkeypatch.setattr(guards, "guard_wifi_location", fake_wifi)
    monkeypatch.setattr(guards, "guard_step_distance", fake_step)
    monkeypatch.setattr(guards, "guard_hr_activity", fake_hr)

    # Run the dispatcher
    result = guards.run_all_guards(
        device_name="phone",
        event_name="wifi_connect",
        event_payload=payload,
        persona_state=persona
    )

    # All three guards should have been invoked
    assert all(called.values()), "Not all guards were called"

    # The payload should still contain the original keys (plus any modifications)
    assert "ssid" in result
    assert "rssi" in result
    # Timestamp should be unchanged because there was no previous timestamp for this device
    assert result["timestamp"] == "2025-12-21T08:00:00+00:00"