# src/utils/logger.py
"""
Utility module that provides a simple, consistently‑formatted logger
for the entire Noise‑Orchestra codebase.

Usage
-----

>>> from src.utils.logger import get_logger
>>> log = get_logger(__name__)          # module‑level logger
>>> log.info("Toolkit started")
>>> log.debug("Debug details …")

The logger writes to **STDOUT** (or STDERR for warnings/errors) with a
human‑readable timestamp and, optionally, can be switched to JSON output
by setting the environment variable ``NOISE_ORCHESTRA_LOG_FORMAT=json``.

Configuration (environment variables)
------------------------------------
* ``NOISE_ORCHESTRA_LOG_LEVEL`` – one of ``DEBUG``, ``INFO``,
  ``WARNING``, ``ERROR`` or ``CRITICAL``.  Defaults to ``INFO``.
* ``NOISE_ORCHESTRA_LOG_FORMAT`` – ``text`` (default) or ``json``.
"""

import json
import logging
import os
import sys
from datetime import datetime
from typing import Optional

# ----------------------------------------------------------------------
# Helper – JSON formatter (optional)
# ----------------------------------------------------------------------
class JsonLogFormatter(logging.Formatter):
    """
    Serialises a LogRecord as a single‑line JSON object.
    Fields included:
        timestamp (ISO‑8601 UTC)
        level
        logger (name)
        message
        any extra key/value pairs passed via ``logger.info(..., extra={...})``
    """

    def format(self, record: logging.LogRecord) -> str:
        log_entry = {
            "timestamp": datetime.utcfromtimestamp(record.created).isoformat() + "Z",
            "level": record.levelname,
            "logger": record.name,
            "message": record.getMessage(),
        }
        # Include any user‑supplied extra attributes (skip internal attrs)
        extra_attrs = {
            key: value
            for key, value in record.__dict__.items()
            if key not in (
                "args", "asctime", "created", "exc_info", "exc_text",
                "filename", "funcName", "levelname", "levelno", "lineno",
                "module", "msecs", "msg", "pathname", "process",
                "processName", "relativeCreated", "stack_info", "thread",
                "threadName"
            )
        }
        if extra_attrs:
            log_entry.update(extra_attrs)

        return json.dumps(log_entry, separators=(",", ":"))

# ----------------------------------------------------------------------
# Helper – plain‑text formatter (default)
# ----------------------------------------------------------------------
class TextLogFormatter(logging.Formatter):
    """
    Human‑readable log format:

        2025-12-21 15:42:07,123 INFO  src.main – Toolkit started
    """
    def __init__(self):
        super().__init__(
            fmt="%(asctime)s %(levelname)-8s %(name)s – %(message)s",
            datefmt="%Y-%m-%d %H:%M:%S"
        )

# ----------------------------------------------------------------------
# Factory – returns a configured logger instance
# ----------------------------------------------------------------------
def get_logger(name: str, level: Optional[int] = None) -> logging.Logger:
    """
    Retrieve (or create) a logger with the project's standard handlers.

    :param name: Logger name – typically ``__name__``.
    :param level: Optional explicit logging level (overrides the env var).
    :return: Configured ``logging.Logger`` instance.
    """
    logger = logging.getLogger(name)

    # Avoid adding duplicate handlers if ``get_logger`` is called repeatedly
    if getattr(logger, "_noise_orchestra_configured", False):
        return logger

    # ------------------------------------------------------------------
    # Determine log level
    # ------------------------------------------------------------------
    env_level = os.getenv("NOISE_ORCHESTRA_LOG_LEVEL", "INFO").upper()
    level_name = level if level is not None else getattr(logging, env_level, logging.INFO)
    logger.setLevel(level_name)

    # ------------------------------------------------------------------
    # Choose formatter (text vs. JSON)
    # ------------------------------------------------------------------
    fmt_choice = os.getenv("NOISE_ORCHESTRA_LOG_FORMAT", "text").lower()
    if fmt_choice == "json":
        formatter = JsonLogFormatter()
    else:
        formatter = TextLogFormatter()

    # ------------------------------------------------------------------
    # Handler – stream to stdout (INFO+) and stderr (WARNING+)
    # ------------------------------------------------------------------
    # INFO and below → stdout
    stdout_handler = logging.StreamHandler(stream=sys.stdout)
    stdout_handler.setLevel(logging.DEBUG)  # let logger filter later
    stdout_handler.addFilter(lambda r: r.levelno < logging.WARNING)
    stdout_handler.setFormatter(formatter)

    # WARNING and above → stderr
    stderr_handler = logging.StreamHandler(stream=sys.stderr)
    stderr_handler.setLevel(logging.WARNING)
    stderr_handler.setFormatter(formatter)

    logger.addHandler(stdout_handler)
    logger.addHandler(stderr_handler)

    # Prevent log propagation to the root logger (avoids duplicate prints)
    logger.propagate = False

    # Mark as configured so repeated calls are cheap
    logger._noise_orchestra_configured = True

    return logger