# src/utils/helpers.py
"""
Utility helpers used across the Noise‑Orchestra codebase.

The functions here are deliberately small, dependency‑free, and have
type hints so they can be imported anywhere without pulling in heavy
libraries.

Commonly used helpers:
* ``haversine_distance`` – great‑circle distance between two lat/lon pairs.
* ``jitter_seconds`` – random small time offset (used for realistic delays).
* ``weighted_choice`` – pick an element from a list of (item, weight) tuples.
* ``format_bytes`` – human‑readable byte‑size string (e.g. “12.3 KB”).
"""

import math
import random
from typing import Iterable, List, Tuple, TypeVar, Callable, Any, Optional

T = TypeVar("T")


# ----------------------------------------------------------------------
# 1️⃣ Haversine distance (metres)
# ----------------------------------------------------------------------
def haversine_distance(coord1: Tuple[float, float],
                       coord2: Tuple[float, float],
                       radius: float = 6371_000) -> float:
    """
    Compute the great‑circle distance between two points on Earth.

    :param coord1: (lat1, lon1) in decimal degrees.
    :param coord2: (lat2, lon2) in decimal degrees.
    :param radius: Earth radius in metres (default 6 371 km).
    :return: Distance in metres (float).
    """
    lat1, lon1 = coord1
    lat2, lon2 = coord2

    # Convert degrees → radians
    phi1 = math.radians(lat1)
    phi2 = math.radians(lat2)
    d_phi = math.radians(lat2 - lat1)
    d_lambda = math.radians(lon2 - lon1)

    a = math.sin(d_phi / 2) ** 2 + \
        math.cos(phi1) * math.cos(phi2) * math.sin(d_lambda / 2) ** 2
    c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))

    return radius * c


# ----------------------------------------------------------------------
# 2️⃣ Jitter – small random offset for timestamps / delays
# ----------------------------------------------------------------------
def jitter_seconds(min_seconds: float = 0.0, max_seconds: float = 2.0) -> float:
    """
    Return a random float between ``min_seconds`` and ``max_seconds``.
    Used to add realistic, non‑deterministic pauses between actions.

    :param min_seconds: Lower bound (inclusive).
    :param max_seconds: Upper bound (exclusive).
    :return: Random delay in seconds.
    """
    return random.uniform(min_seconds, max_seconds)


# ----------------------------------------------------------------------
# 3️⃣ Weighted random choice
# ----------------------------------------------------------------------
def weighted_choice(items: Iterable[Tuple[T, float]]) -> T:
    """
    Choose a single element from ``items`` where each tuple is
    (value, weight).  Weights must be non‑negative; zero‑weight items
    will never be selected.

    Example:
        >>> options = [("low", 3), ("mid", 2), ("high", 1)]
        >>> weighted_choice(options)   # probabilistically returns one of the strings
    """
    population = []
    cum_weights = []
    total = 0.0
    for value, weight in items:
        if weight <= 0:
            continue
        total += weight
        population.append(value)
        cum_weights.append(total)

    if not population:
        raise ValueError("weighted_choice received an empty or zero‑weight iterable")

    rnd = random.uniform(0, total)
    # Linear search – fine for the tiny lists we use (typically < 10 items)
    for idx, cw in enumerate(cum_weights):
        if rnd <= cw:
            return population[idx]

    # Fallback (should never happen because rnd <= total)
    return population[-1]


# ----------------------------------------------------------------------
# 4️⃣ Human‑readable byte formatting (optional convenience)
# ----------------------------------------------------------------------
def format_bytes(num_bytes: float, precision: int = 2) -> str:
    """
    Convert a byte count into a friendly string (KB, MB, GB, …).

    :param num_bytes: Size in bytes.
    :param precision: Decimal places for the formatted number.
    :return: String like “12.34 KB”.
    """
    if num_bytes < 0:
        raise ValueError("num_bytes must be non‑negative")

    units = ["B", "KB", "MB", "GB", "TB", "PB"]
    idx = 0
    while num_bytes >= 1024 and idx < len(units) - 1:
        num_bytes /= 1024.0
        idx += 1
    return f"{num_bytes:.{precision}f}\u202F{units[idx]}"  # \u202F = narrow no‑break space


# ----------------------------------------------------------------------
# 5️⃣ Simple memoization decorator (optional – used by some API clients)
# ----------------------------------------------------------------------
def memoize(ttl_seconds: int = 300):
    """
    Very small in‑memory memoizer.  Stores the result of a function call
    for ``ttl_seconds`` seconds.  Only works for functions with hashable
    positional arguments (no **kwargs handling).

    Example usage:

        @memoize(ttl_seconds=600)
        def expensive_lookup(arg1, arg2):
            ...

    """
    def decorator(func: Callable):
        cache: dict = {}

        def wrapper(*args):
            now = datetime.utcnow().timestamp()
            key = (func, args)
            if key in cache:
                result, expires = cache[key]
                if now < expires:
                    return result
                else:
                    del cache[key]  # expired
            result = func(*args)
            cache[key] = (result, now + ttl_seconds)
            return result

        return wrapper
    return decorator