# src/main.py
import argparse
import os
import sys
import yaml
from pathlib import Path
from src.persona import Persona
from src.itinerary import ItineraryBuilder
from src.events import EventEmitter
from src.utils.logger import get_logger

log = get_logger(__name__)

def load_yaml(path: Path):
    with open(path, "r", encoding="utf-8") as f:
        return yaml.safe_load(f)

def parse_args():
    parser = argparse.ArgumentParser(
        description="Noise‑Orchestra – generate multi‑device privacy noise"
    )
    parser.add_argument("--persona", type=Path, default=Path("config/persona.yaml"))
    parser.add_argument("--destinations", type=Path, default=Path("config/destinations.yaml"))
    parser.add_argument("--airports", type=Path, default=Path("config/airports.yaml"))
    parser.add_argument("--log-file", type=Path, default=None,
                        help="Write JSON event stream to this file (default: stdout)")
    parser.add_argument("--speed-factor", type=float, default=0.001,
                        help="Accelerate simulated time (e.g., 0.001 = 1 h runs in 3.6 s)")
    parser.add_argument("--max-clones", type=int, default=1,
                        help="Number of concurrent clones to run")
    parser.add_argument("--debug", action="store_true")
    return parser.parse_args()

def main():
    args = parse_args()
    if args.debug:
        log.setLevel("DEBUG")

    # Load configuration files
    persona_cfg = load_yaml(args.persona)
    destinations_cfg = load_yaml(args.destinations)
    airports_cfg = load_yaml(args.airports)

    # Prepare output sink
    if args.log_file:
        sink = open(args.log_file, "w", encoding="utf-8")
    else:
        sink = sys.stdout

    # Run the requested number of clones sequentially (you can parallelise later)
    for clone_idx in range(args.max_clones):
        log.info(f"=== Starting clone #{clone_idx + 1} ===")
        # 1️⃣ Initialise persona (shared state for phone/laptop/watch)
        persona = Persona(persona_cfg)

        # 2️⃣ Build an itinerary (destination + flight info)
        itinerary = ItineraryBuilder(destinations_cfg, airports_cfg, persona)
        itinerary.plan_next_leg()   # selects destination, fetches flight data, etc.

        # 3️⃣ Event emitter (writes JSON lines)
        emitter = EventEmitter(persona, sink, speed_factor=args.speed_factor)

        # 4️⃣ Run the full day simulation
        emitter.run_full_day(itinerary)

        log.info(f"=== Clone #{clone_idx + 1} finished ===")

    if args.log_file:
        sink.close()

if __name__ == "__main__":
    main()

🗒️ 3️⃣ src/persona.py

# src/persona.py
import random
import uuid
import datetime as dt
import pytz
from src.utils.helpers import jitter_seconds

class Persona:
    """
    Holds the *single source of truth* for a clone:
    - fingerprint (UA, language, screen, plugins)
    - biometric baselines (HR, steps)
    - battery levels per device
    - current geographic state (lat, lon, tz)
    - shared cookie jar
    """

    def __init__(self, cfg: dict):
        # ==== Fingerprint ====
        self.ua = {
            "phone":   cfg["user_agents"]["phone"],
            "laptop":  cfg["user_agents"]["laptop"],
            "watch":   cfg["user_agents"]["watch"]
        }
        self.language = cfg.get("language", "en-US")
        self.screen_resolution = cfg.get("screen_resolution", "1920x1080")
        self.color_depth = cfg.get("color_depth", 24)
        self.plugins = cfg.get("plugins", [])
        self.canvas_hash = cfg.get("canvas_hash")
        self.webgl_renderer = cfg.get("webgl_renderer")
        self.referrer_policy = cfg.get("referrer_policy")
        self.cache_control = cfg.get("cache_control")
        self.navigator = cfg.get("navigator")

        # ==== Biometric baselines ====
        self.baseline_hr = cfg.get("baseline_heart_rate", 72)
        self.current_hr = {
            "phone": self.baseline_hr,
            "laptop": self.baseline_hr,
            "watch": self.baseline_hr
        }
        self.step_counter = 0
        self.stride_m = 0.78   # average stride in metres

        # ==== Battery ====
        cap = cfg.get("battery_capacity", {})
        self.battery = {
            "phone": cap.get("phone", 100),
            "laptop": cap.get("laptop", 100),
            "watch": cap.get("watch", 100)
        }

        # ==== Location ====
        # start at a neutral point (home) – will be overwritten by itinerary
        self.lat = 0.0
        self.lon = 0.0
        self.tz = "UTC"

        # ==== Cookie jar (shared) ====
        self.cookie_jar = {}

        # ==== Misc ====
        self.id = str(uuid.uuid4())
        self.assisted = False   # set to True for blind‑travel assistance

    # -----------------------------------------------------------------
    # Helpers that devices call to read/write shared state
    # -----------------------------------------------------------------
    def set_location(self, lat: float, lon: float, tz_name: str):
        self.lat = lat
        self.lon = lon
        self.tz = tz_name

    def get_timestamp(self) -> str:
        """Current time in the persona's local timezone (ISO‑8601)."""
        now = dt.datetime.now(pytz.timezone(self.tz))
        return now.isoformat()

    def add_steps(self, count: int):
        self.step_counter += count

    def update_heart_rate(self, device: str, delta: int):
        """Adjust HR for a device; keep it within realistic bounds."""
        new_hr = self.current_hr[device] + delta
        new_hr = max(50, min(190, new_hr))   # clamp
        self.current_hr[device] = new_hr
        return new_hr

    def drain_battery(self, device: str, amount: float):
        """Subtract amount (percent). Never go below 0."""
        self.battery[device] = max(0, self.battery[device] - amount)

    def charge_battery(self, device: str, amount: float):
        self.battery[device] = min(100, self.battery[device] + amount)

    # -----------------------------------------------------------------
    # Clipboard handling (shared across devices)
    # -----------------------------------------------------------------
    def copy_to_clipboard(self, source_device: str, content_type: str, content: str, description: str = ""):
        entry = {
            "source_device": source_device,
            "content_type": content_type,
            "content": content,
            "description": description,
            "timestamp": self.get_timestamp()
        }
        self.cookie_jar["clipboard_latest"] = entry
        return entry

    def get_clipboard(self):
        return self.cookie_jar.get("clipboard_latest")