# src/itinerary.py
import random
import datetime as dt
import pytz
from src.api_clients.aviationstack import get_flight_info
from src.api_clients.wettr import get_temperature
from src.utils.helpers import weighted_choice

class ItineraryBuilder:
    """
    Takes the destination pool, airport metadata and the persona,
    then selects a realistic next leg (airport → destination → flight).
    """

    def __init__(self, destinations_cfg, airports_cfg, persona):
        self.destinations = destinations_cfg      # list of dicts
        self.airports = airports_cfg              # dict keyed by IATA
        self.persona = persona
        self.current_airport_iata = None
        self.next_flight = None
        self.destination = None

    # -----------------------------------------------------------------
    def _determine_home_season(self):
        """Uses wettr to decide if we are in winter / summer / shoulder."""
        home_city = self.persona.id.split("-")[0]   # placeholder; replace with real home city if needed
        temp_c = get_temperature(home_city)        # returns int Celsius
        if temp_c <= 10:
            return "winter"
        elif temp_c >= 30:
            return "summer"
        else:
            return "shoulder"

    # -----------------------------------------------------------------
    def _filter_destinations(self, season):
        """Keep only entries that match the season or are marked 'any'."""
        return [
            d for d in self.destinations
            if d["season"] in (season, "any")
        ]

    # -----------------------------------------------------------------
    def plan_next_leg(self):
        """Select destination, fetch flight, and update persona location."""
        season = self._determine_home_season()
        candidates = self._filter_destinations(season)

        # Apply budget weighting (low=3, mid=2, high=1)
        weighted = []
        for d in candidates:
            weight = {"low": 3, "mid": 2, "high": 1}[d["budget"]]
            weighted.append((d, weight))

        self.destination = weighted_choice(weighted)   # returns the dict
        dest_iata = self.destination["iata"]

        # Choose a departure airport (for simplicity pick a random one from the pool)
        self.current_airport_iata = random.choice(list(self.airports.keys()))
        dep_info = self.airports[self.current_airport_iata]

        # Update persona location to the departure airport
        self.persona.set_location(dep_info["lat"], dep_info["lon"], dep_info["tz"])

        # Pull a real flight that matches the route (today + 1‑2 days)
        today = dt.date.today()
        for offset in range(1, 4):   # try next 3 days
            flight_date = today + dt.timedelta(days=offset)
            flight = get_flight_info(
                dep_iata=self.current_airport_iata,
                arr_iata=dest_iata,
                date=str(flight_date)
            )
            if flight:
                self.next_flight = flight
                break

        if not self.next_flight:
            raise RuntimeError("Could not find a real flight for the chosen route.")

        # Store flight metadata on the persona for later use
        self.persona.next_flight = self.next_flight
        self.persona.destination_airport = self.airports[dest_iata]

        # Log a brief summary (optional)
        print(f"[Itinerary] {self.current_airport_iata} → {dest_iata} "
              f"flight {self.next_flight['flight_number']} on {self.next_flight['date']} "
              f"(delay {self.next_flight.get('delay',0)} min)")

    # -----------------------------------------------------------------
    def get_current_leg_info(self):
        """Convenient accessor for the emitter."""
        return {
            "departure_airport": self.current_airport_iata,
            "destination_airport": self.destination["iata"],
            "flight": self.next_flight,
            "destination_meta": self.persona.destination_airport
        }