# src/events.py
import json
import time
from src.utils.helpers import jitter_seconds

class EventEmitter:
    """
    Central place that turns persona state changes into JSON lines.
    All three devices (phone, laptop, watch) emit the *same* timestamp
    for a given logical event, guaranteeing perfect synchronization.
    """

    def __init__(self, persona, sink, speed_factor=0.001):
        """
        :param persona: Persona instance (shared state)
        :param sink: file‑like object (stdout or opened file)
        :param speed_factor: accelerate simulated time (real_seconds * speed_factor = sleep)
        """
        self.persona = persona
        self.sink = sink
        self.speed_factor = speed_factor

    # -----------------------------------------------------------------
    def _write_event(self, event_dict):
        line = json.dumps(event_dict, ensure_ascii=False)
        self.sink.write(line + "\n")
        self.sink.flush()

    # -----------------------------------------------------------------
    def _emit_common(self, device, event_name, extra=None):
        payload = {
            "device": device,
            "event": event_name,
            "timestamp": self.persona.get_timestamp(),
            "location": {"lat": self.persona.lat, "lon": self.persona.lon},
            "timezone": self.persona.tz,
            "user_agent": self.persona.ua[device],
            "language": self.persona.language,
            "battery_percent": self.persona.battery[device],
            "heart_rate": self.persona.current_hr[device],
        }
        if extra:
            payload.update(extra)
        self._write_event(payload)

    # -----------------------------------------------------------------
    # Example high‑level actions (add more as needed)

    def emit_wifi_connect(self, device, ssid, bssid, rssi):
        self._emit_common(
            device,
            "wifi_connect",
            {"ssid": ssid, "bssid": bssid, "rssi": rssi}
        )

    def emit_wifi_disconnect(self, device, ssid):
        self._emit_common(
            device,
            "wifi_disconnect",
            {"ssid": ssid}
        )

    def emit_http_get(self, device, url, size_kb):
        self._emit_common(
            device,
            "http_get",
            {"url": url, "size_kb": size_kb}
        )

    def emit_clipboard_sync(self, source_device, content_type, content, description=""):
        """
        Broadcast a clipboard copy to all devices.
        """
        # Store the latest clipboard entry in the shared persona
        self.persona.copy_to_clipboard(
            source_device, content_type, content, description
        )
        for dev in ("phone", "laptop", "watch"):
            self._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": source_device,
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )

    def emit_clipboard_paste(self, device, content_type, content, action="open_browser_tab"):
        """
        Log when a device actually uses the clipboard content.
        """
        self._emit_common(
            device,
            "clipboard_paste",
            {
                "content_type": content_type,
                "content": content,
                "action": action
            }
        )

    def emit_seatbelt_announcement(self, device):
        """
        Simulate the seat‑belt alert: short HR bump + activity change,
        then normalize after ~60 seconds.
        """
        # HR bump
        self.persona.update_heart_rate(device, +5)
        self._emit_common(
            device,
            "seatbelt_announcement",
            {"hr": self.persona.current_hr[device], "activity_state": "alerted"}
        )
        # Wait (accelerated) then normalize
        time.sleep(60 * self.speed_factor)
        self.persona.update_heart_rate(device, -5)
        self._emit_common(
            device,
            "hr_normalization",
            {"hr": self.persona.current_hr[device], "activity_state": "seated"}
        )

    # -----------------------------------------------------------------
    # Core simulation loop (high‑level – expand per your itinerary)

    def run_full_day(self, itinerary_builder):
        """
        Executes a simplified day:
        1. Airport arrival → Wi‑Fi connect
        2. Security (with assisted wait)
        3. Flight‑delay alert (if any)
        4. In‑flight seat‑belt announcement
        5. Clipboard sync of flight‑status URL
        6. Landing → hotel Wi‑Fi connect, clipboard paste of hotel link
        7. End of day → device stabilisation
        """
        leg = itinerary_builder.get_current_leg_info()
        dep_airport = leg["departure_airport"]
        dest_airport = leg["destination_airport"]
        flight = leg["flight"]

        # ---- 1️⃣ Airport arrival (phone + watch stay on Wi‑Fi) ----
        ssid = random.choice(dep_airport["wifi_ssids"])
        bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        rssi = random.randint(-70, -40)
        for dev in ("phone", "watch"):
            self.emit_wifi_connect(dev, ssid, bssid, rssi)

        # ---- 2️⃣ Security checkpoint (assistive wait) ----
        base_wait = dep_airport.get("security_base_wait_sec", 600)   # 10 min default
        multiplier = random.uniform(1.20, 1.40)   # blind‑travel assistance
        wait_sec = int(base_wait * multiplier)
        self._emit_common(
            "phone",
            "security_check_start",
            {"airport_id": dep_airport["airport_id"], "estimated_wait_sec": wait_sec}
        )
        time.sleep(wait_sec * self.speed_factor)
        self._emit_common(
            "phone",
            "security_check_end",
            {"airport_id": dep_airport["airport_id"], "total_elapsed_sec": wait_sec}
        )

        # ---- 3️⃣ Flight‑delay alert (if applicable) ----
        delay = flight.get("delay", 0)
        if delay:
            msg = f"Flight {flight['flight_number']} delayed {delay} min."
            for dev in ("phone", "watch"):
                self._emit_common(
                    dev,
                    "flight_delay_alert",
                    {"flight_number": flight["flight_number"], "delay_minutes": delay, "message": msg}
                )

        # ---- 4️⃣ In‑flight seat‑belt announcement ----
        for dev in ("watch",):
            self.emit_seatbelt_announcement(dev)

        # ---- 5️⃣ Clipboard sync of flight‑status URL ----
        status_url = flight.get("status_url", f"https://airline.example.com/flight/{flight['flight_number']}")
        self.emit_clipboard_sync("phone", "url", status_url, "Flight status page")

        # ---- 6️⃣ Landing → hotel Wi‑Fi connect ----
        hotel_ssid = random.choice(dest_airport.get("hotel_wifi_ssids", ["Hotel_Free_WiFi"]))
        bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        rssi = random.randint(-70, -40)
        for dev in ("phone", "laptop", "watch"):
            self.emit_wifi_connect(dev, hotel_ssid, bssid, rssi)

        # Paste the hotel reservation link (simulated)
        hotel_url = f"https://booking.example.com/hotel/{dest_airport['airport_id']}"
        self.emit_clipboard_sync("phone", "url", hotel_url, "Hotel reservation")
        self.emit_clipboard_paste("laptop", "url", hotel_url, "open_browser_tab")

        # ---- 7️⃣ End of day (stabilisation) ----
        for dev in ("phone", "laptop", "watch"):
            self._emit_common(
                dev,
                "device_stabilization",
                {"note": "End of simulated day"}
            )