# src/devices/watch.py
"""
Smartwatch device simulation.

The ``Watch`` class mirrors a wearable that a traveler would wear
continuously: it tracks steps, heart‑rate, accelerometer data, handles
clipboard synchronization, and can emit short “heartbeat‑monitor” packets
while the user is stationary (e.g., on a conveyor belt or in a security
line).

All actions are routed through the shared ``Persona`` instance so the
phone, laptop, and watch stay perfectly synchronized.  Events are
emitted via the ``EventEmitter`` passed at construction time.
"""

import logging
import random
import time
from datetime import datetime, timedelta

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Tiny jitter helper – makes timing look human‑like
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class Watch:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.connected_ssid = None
        self.last_step_timestamp = None   # for step‑rate calculations

    # ------------------------------------------------------------------
    # Wi‑Fi handling (mirrors Phone/Laptop)
    # ------------------------------------------------------------------
    def connect_wifi(self, ssid: str, bssid: str = None, rssi: int = None):
        """
        Simulate a Wi‑Fi association for the watch.
        """
        if not bssid:
            bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        if not rssi:
            rssi = random.randint(-70, -40)

        self.connected_ssid = ssid
        self.emitter.emit_wifi_connect("watch", ssid, bssid, rssi)
        time.sleep(_jitter() * self.speed_factor)

    def disconnect_wifi(self):
        """Simulate leaving the current network."""
        if self.connected_ssid:
            self.emitter.emit_wifi_disconnect("watch", self.connected_ssid)
            self.connected_ssid = None
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Step & heart‑rate simulation
    # ------------------------------------------------------------------
    def _emit_step_event(self, steps: int):
        """
        Emit a step_increment event and update the persona’s step counter.
        Also adjusts heart‑rate according to the steps‑↔‑HR coupling.
        """
        self.persona.add_steps(steps)
        # Update HR based on steps (5 steps ≈ 1 bpm increase, as defined in guards)
        hr_change = steps // 30   # integer division – 30 steps ≈ +1 bpm
        if hr_change:
            new_hr = self.persona.update_heart_rate("watch", hr_change)
        else:
            new_hr = self.persona.current_hr["watch"]

        self.emitter._emit_common(
            "watch",
            "step_increment",
            {
                "steps": steps,
                "heart_rate": new_hr,
                "activity_state": "walking"
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    def simulate_walk(self, duration_seconds: int, avg_steps_per_min: int = 100):
        """
        Simulate a walking segment.  The function divides the total duration
        into 1‑minute chunks (or shorter if the duration is < 60 s) and emits
        step_increment events for each chunk.

        :param duration_seconds: Total walking time.
        :param avg_steps_per_min: Approximate cadence (default 100 spm).
        """
        remaining = duration_seconds
        while remaining > 0:
            chunk = min(60, remaining)   # 1‑minute granularity
            steps_this_chunk = int(avg_steps_per_min * (chunk / 60.0))
            self._emit_step_event(steps_this_chunk)
            remaining -= chunk

    def simulate_stationary(self, duration_seconds: int = 30):
        """
        While the wearer is still (e.g., waiting in a security line or
        sitting on the conveyor belt), emit a small “heartbeat‑monitor”
        packet every ~10 seconds to prove the device is still being worn.

        The watch’s HR will *gradually* drift back toward the baseline
        (handled by the HR‑step guard, but we also emit a tiny decrement
        here for realism).
        """
        intervals = max(1, duration_seconds // 10)
        for _ in range(intervals):
            # Small HR decay (‑1 bpm) to simulate calming down
            self.persona.update_heart_rate("watch", -1)
            self.emitter._emit_common(
                "watch",
                "heartbeat_monitor",
                {
                    "heart_rate": self.persona.current_hr["watch"],
                    "activity_state": "seated"
                }
            )
            time.sleep(10 * self.speed_factor)

    # ------------------------------------------------------------------
    # Accelerometer simulation (used for conveyor‑belt / elevator detection)
    # ------------------------------------------------------------------
    def emit_accelerometer(self, x: float, y: float, z: float):
        """
        Emit a raw accelerometer reading.  Values are in g‑units; typical
        stationary reading is (0.0, 0.0, 1.0).  Small jitter is added to
        make the data look natural.
        """
        jitter = lambda: random.uniform(-0.02, 0.02)
        payload = {
            "x": round(x + jitter(), 3),
            "y": round(y + jitter(), 3),
            "z": round(z + jitter(), 3)
        }
        self.emitter._emit_common("watch", "accelerometer", payload)
        time.sleep(_jitter() * self.speed_factor)

    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        The watch sits on the X‑ray belt: Wi‑Fi stays connected, but the
        accelerometer reads near‑zero movement.  We emit a series of
        stationary accelerometer events and a heartbeat monitor.
        """
        # Keep Wi‑Fi as‑is (no disconnect)
        for _ in range(duration_seconds):
            self.emit_accelerometer(0.0, 0.0, 0.0)   # essentially no motion
            self.emitter._emit_common(
                "watch",
                "heartbeat_monitor",
                {
                    "heart_rate": self.persona.current_hr["watch"],
                    "activity_state": "seated"
                }
            )
            time.sleep(1 * self.speed_factor)

    # ------------------------------------------------------------------
    # Clipboard handling (mirrors Phone/Laptop)
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copy something to the watch’s clipboard and broadcast the sync to
        all devices via the shared Persona.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="watch",
            content_type=content_type,
            content=content,
            description=description
        )
        for dev in ("phone", "laptop", "watch"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "watch",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "open_browser_tab"):
        """
        Simulate the user pasting the current clipboard content.
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("Watch paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "watch",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Battery handling (mirrors other devices)
    # ------------------------------------------------------------------
    def drain_battery(self, amount: float):
        """
        Decrease battery level by ``amount`` percent (used during active
        streaming or heavy sensor usage).
        """
        self.persona.drain_battery("watch", amount)
        self.emitter._emit_common(
            "watch",
            "battery_update",
            {"battery_percent": self.persona.battery["watch"]}
        )
        time.sleep(_jitter() * self.speed_factor)

    def charge_battery(self, amount: float):
        """
        Increase battery level by ``amount`` percent (used when the watch
        is placed on a charging pad).
        """
        self.persona.charge_battery("watch", amount)
        self.emitter._emit_common(
            "watch",
            "battery_update",
            {"battery_percent": self.persona.battery["watch"]}
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Seat‑belt alert (mirrors the generic function in EventEmitter)
    # ------------------------------------------------------------------
    def seatbelt_alert(self):
        """
        Shortcut that forwards to the central EventEmitter implementation.
        """
        self.emitter.emit_seatbelt_announcement("watch")