# src/devices/phone.py
"""
Phone device simulation.

The `Phone` class is a thin wrapper around the shared ``Persona`` state.
It knows how to:

* Connect / disconnect from Wi‑Fi (emits the appropriate events).
* Perform HTTP GET/POST requests (used for weather, flight‑status, etc.).
* Handle clipboard operations (copy → sync, paste).
* Emit voice‑assistant queries (STT upload, intent routing, TTS download).
* Simulate a short “device‑lost on conveyor belt” pause (used by the
  security‑checkpoint guard).

All actions ultimately call ``EventEmitter._emit_common`` via the
``event_emitter`` instance that is passed in at construction time.
"""

import json
import logging
import random
import time
from urllib.parse import urlencode

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Helper – tiny random jitter for realistic timing
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class Phone:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.connected_ssid = None

    # ------------------------------------------------------------------
    # Wi‑Fi handling
    # ------------------------------------------------------------------
    def connect_wifi(self, ssid: str, bssid: str = None, rssi: int = None):
        """
        Simulate a Wi‑Fi association.  If ``bssid`` or ``rssi`` are not
        supplied we generate plausible values.
        """
        if not bssid:
            bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        if not rssi:
            rssi = random.randint(-70, -40)

        self.connected_ssid = ssid
        self.emitter.emit_wifi_connect("phone", ssid, bssid, rssi)
        time.sleep(_jitter() * self.speed_factor)

    def disconnect_wifi(self):
        """Simulate leaving the current network."""
        if self.connected_ssid:
            self.emitter.emit_wifi_disconnect("phone", self.connected_ssid)
            self.connected_ssid = None
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Generic HTTP request helpers (used by many higher‑level actions)
    # ------------------------------------------------------------------
    def _http(self, method: str, url: str, **kwargs):
        """
        Low‑level wrapper that performs the request, logs size, and emits
        the appropriate event (http_get / http_post).  ``kwargs`` are passed
        straight to ``requests.<method>``.
        """
        headers = kwargs.pop("headers", {})
        # Ensure the phone’s UA is sent
        headers["User-Agent"] = self.persona.ua["phone"]
        # Optional referrer (useful for simulating navigation chains)
        if "referrer" in kwargs:
            headers["Referer"] = kwargs.pop("referrer")

        try:
            resp = requests.request(method, url, headers=headers, timeout=10, **kwargs)
            resp.raise_for_status()
            size_kb = len(resp.content) / 1024.0
            event_name = "http_get" if method.upper() == "GET" else "http_post"
            self.emitter._emit_common(
                "phone",
                event_name,
                {"url": url, "size_kb": round(size_kb, 2)}
            )
            # Simulate network latency
            time.sleep(_jitter() * self.speed_factor)
            return resp
        except Exception as exc:
            log.warning(f"Phone HTTP {method} to {url} failed: {exc}")
            return None

    def http_get(self, url: str, **kwargs):
        return self._http("GET", url, **kwargs)

    def http_post(self, url: str, data=None, json_data=None, **kwargs):
        if json_data is not None:
            kwargs["json"] = json_data
        else:
            kwargs["data"] = data
        return self._http("POST", url, **kwargs)

    # ------------------------------------------------------------------
    # Clipboard handling (delegates to Persona)
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copies something to the phone’s clipboard and triggers a sync
        across all devices via the Persona.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="phone",
            content_type=content_type,
            content=content,
            description=description
        )
        # Emit the sync event for each device (phone, laptop, watch)
        for dev in ("phone", "laptop", "watch"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "phone",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "open_browser_tab"):
        """
        Simulate the user pasting the current clipboard content.
        The action string is free‑form – e.g. "open_browser_tab",
        "read_aloud", "share_via_message".
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("Phone paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "phone",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Voice‑assistant interaction
    # ------------------------------------------------------------------
    def voice_query(self, query: str, language: str = None):
        """
        Simulate speaking a query to a smart‑assistant (e.g. “Hey Lumo,
        what’s the weather in Tokyo?”).

        The flow:
        1️⃣ Upload the audio (STT) – we fake it with a tiny POST.
        2️⃣ Receive the transcribed text (here we just echo the query).
        3️⃣ Send the intent to the assistant’s NLP endpoint.
        4️⃣ Receive a response (simulated) and optionally download a TTS
           audio snippet.
        """
        # ---- 1️⃣ STT upload (tiny placeholder payload) ----
        stt_url = "https://stt.lumo.me/v1/recognize"
        stt_payload = {"audio_blob": "base64‑encoded‑placeholder"}  # real impl would send bytes
        self.http_post(stt_url, json_data=stt_payload)
        # ---- 2️⃣ Intent routing ----
        intent_url = "https://api.lumo.me/v1/interpret"
        intent_payload = {"text": query, "language": language or self.persona.language}
        resp = self.http_post(intent_url, json_data=intent_payload)
        # Assume the response contains a field "reply_text"
        reply_text = ""
        if resp and resp.ok:
            try:
                reply_json = resp.json()
                reply_text = reply_json.get("reply_text", "")
            except Exception:
                pass

        # ---- 3️⃣ TTS download (if there is a reply) ----
        if reply_text:
            tts_url = "https://tts.lumo.me/v1/synthesize"
            tts_params = {"text": reply_text, "voice": "en-US"}
            # The TTS endpoint returns an audio file; we just log size.
            tts_resp = self.http_get(tts_url, params=tts_params)
            # Emit a dedicated event so downstream logs can see the audio download
            if tts_resp:
                size_kb = len(tts_resp.content) / 1024.0
                self.emitter._emit_common(
                    "phone",
                    "tts_download",
                    {"url": tts_url, "size_kb": round(size_kb, 2), "text": reply_text}
                )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Simulated “device lost on conveyor belt” (used by security guard)
    # ------------------------------------------------------------------
    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        The phone is placed on the X‑ray belt; Wi‑Fi disconnects,
        then reconnects after ``duration_seconds`` (simulated time).
        """
        # Disconnect (the belt cuts the radio)
        self.disconnect_wifi()
        # Wait the simulated belt time (accelerated)
        time.sleep(duration_seconds * self.speed_factor)
        # Re‑connect to the same SSID if we still have one stored
        if self.persona.current_venue and "wifi_ssids" in self.persona.current_venue:
            # Pick a random SSID from the venue (usually the same as before)
            ssid = random.choice(self.persona.current_venue["wifi_ssids"])
            self.connect_wifi(ssid)
        else:
            # Fallback – just reconnect to a generic free network
            self.connect_wifi("Free_WiFi_Generic")