# src/devices/laptop.py
"""
Laptop device simulation.

The ``Laptop`` class mirrors the behaviour of a typical notebook that a
traveler would carry: it connects to airport / hotel Wi‑Fi, performs HTTP
requests (web browsing, downloading PDFs, streaming short video clips),
participates in clipboard synchronization, and can play media (e.g. an
in‑flight entertainment snippet).

All actions funnel through the shared ``Persona`` instance so the three
devices stay perfectly in sync, and they emit events via the
``EventEmitter`` passed at construction time.
"""

import json
import logging
import random
import time
from urllib.parse import urlencode

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Tiny jitter helper – makes the timing look human‑like
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class Laptop:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.connected_ssid = None

    # ------------------------------------------------------------------
    # Wi‑Fi handling
    # ------------------------------------------------------------------
    def connect_wifi(self, ssid: str, bssid: str = None, rssi: int = None):
        """
        Simulate a Wi‑Fi association for the laptop.
        """
        if not bssid:
            bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        if not rssi:
            rssi = random.randint(-70, -40)

        self.connected_ssid = ssid
        self.emitter.emit_wifi_connect("laptop", ssid, bssid, rssi)
        time.sleep(_jitter() * self.speed_factor)

    def disconnect_wifi(self):
        """Simulate leaving the current network."""
        if self.connected_ssid:
            self.emitter.emit_wifi_disconnect("laptop", self.connected_ssid)
            self.connected_ssid = None
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Generic HTTP request helper (GET / POST)
    # ------------------------------------------------------------------
    def _http(self, method: str, url: str, **kwargs):
        """
        Low‑level wrapper that performs the request, logs size, and emits
        the appropriate event (http_get / http_post).  ``kwargs`` are passed
        straight to ``requests.<method>``.
        """
        headers = kwargs.pop("headers", {})
        # Ensure the laptop’s UA is sent
        headers["User-Agent"] = self.persona.ua["laptop"]
        # Optional referrer (useful for navigation chains)
        if "referrer" in kwargs:
            headers["Referer"] = kwargs.pop("referrer")

        try:
            resp = requests.request(method, url, headers=headers, timeout=10, **kwargs)
            resp.raise_for_status()
            size_kb = len(resp.content) / 1024.0
            event_name = "http_get" if method.upper() == "GET" else "http_post"
            self.emitter._emit_common(
                "laptop",
                event_name,
                {"url": url, "size_kb": round(size_kb, 2)}
            )
            # Simulate network latency
            time.sleep(_jitter() * self.speed_factor)
            return resp
        except Exception as exc:
            log.warning(f"Laptop HTTP {method} to {url} failed: {exc}")
            return None

    def http_get(self, url: str, **kwargs):
        return self._http("GET", url, **kwargs)

    def http_post(self, url: str, data=None, json_data=None, **kwargs):
        if json_data is not None:
            kwargs["json"] = json_data
        else:
            kwargs["data"] = data
        return self._http("POST", url, **kwargs)

    # ------------------------------------------------------------------
    # Clipboard handling (mirrors Phone)
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copy something to the laptop’s clipboard and broadcast the sync to
        all devices via the shared Persona.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="laptop",
            content_type=content_type,
            content=content,
            description=description
        )
        for dev in ("phone", "laptop", "watch"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "laptop",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "open_browser_tab"):
        """
        Simulate pasting the current clipboard content.
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("Laptop paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "laptop",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Media streaming – used for in‑flight entertainment (IFE) snippets
    # ------------------------------------------------------------------
    def stream_media(self, media_url: str, chunk_size_kb: int = 200, max_chunks: int = 3):
        """
        Simulate streaming a short video/audio clip (e.g. a 30‑second
        in‑flight entertainment segment).  The function performs a series
        of GET requests with a ``Range`` header to mimic chunked
        downloading.

        :param media_url: Base URL of the media file.
        :param chunk_size_kb: Approximate size of each chunk.
        :param max_chunks: Number of chunks to fetch (controls total bandwidth).
        """
        for i in range(max_chunks):
            start = i * chunk_size_kb * 1024
            end = start + chunk_size_kb * 1024 - 1
            headers = {"Range": f"bytes={start}-{end}"}
            resp = self.http_get(media_url, headers=headers, referrer=None)
            if not resp:
                break   # stop if a chunk fails
            # Emit a dedicated streaming‑chunk event (useful for analytics)
            self.emitter._emit_common(
                "laptop",
                "media_stream_chunk",
                {
                    "url": media_url,
                    "chunk_index": i,
                    "size_kb": chunk_size_kb,
                    "byte_range": f"{start}-{end}"
                }
            )
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # File download – e.g., hotel reservation PDF, travel guide, etc.
    # ------------------------------------------------------------------
    def download_file(self, file_url: str, dest_path: str = None):
        """
        Perform a GET request for a file (PDF, DOCX, etc.) and emit a
        ``file_download`` event.  ``dest_path`` is optional – the toolkit
        does not actually write to disk (that would be unnecessary for the
        noise model), but the parameter is kept for realism.

        :param file_url: Direct URL to the file.
        :param dest_path: Where the file *would* be saved (ignored).
        """
        resp = self.http_get(file_url, referrer=None)
        if not resp:
            return

        size_kb = len(resp.content) / 1024.0
        self.emitter._emit_common(
            "laptop",
            "file_download",
            {
                "url": file_url,
                "size_kb": round(size_kb, 2),
                "filename": dest_path or file_url.split("/")[-1]
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Simulated “device lost on conveyor belt” (mirrors Phone)
    # ------------------------------------------------------------------
    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        The laptop is placed on the X‑ray belt; Wi‑Fi disconnects,
        then reconnects after ``duration_seconds`` (simulated time).
        """
        self.disconnect_wifi()
        time.sleep(duration_seconds * self.speed_factor)

        # Re‑connect to a venue‑appropriate SSID if we have venue info
        if hasattr(self.persona, "current_venue") and self.persona.current_venue:
            ssid = random.choice(self.persona.current_venue.get("wifi_ssids", ["Free_WiFi_Generic"]))
            self.connect_wifi(ssid)
        else:
            self.connect_wifi("Free_WiFi_Generic")