# src/devices/ipad.py
"""
iPad (tablet) simulation.

The ``IPad`` class represents a consumer tablet that a traveler might
use for browsing, listening to music, or watching short videos while
waiting in an airport lounge.  It supports:

* Wi‑Fi association (same pattern as Phone/Laptop/E‑Reader).
* Bluetooth‑headphone pairing (used for the “in‑flight audio” scenario).
* Music‑playlist streaming – broken into discrete ``media_stream_chunk``
  events so the traffic looks like a real streaming client.
* Clipboard synchronization (copy → paste) – identical to the other
  devices.
* Battery‑drain / charge handling.
* A tiny “conveyor‑belt” helper (the iPad can stay on the belt while the
  watch is still connected).

All actions go through the shared ``Persona`` instance, guaranteeing
perfect cross‑device consistency.
"""

import logging
import random
import time
from urllib.parse import urlencode

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Tiny jitter helper – makes timing look human‑like
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class IPad:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.connected_ssid = None
        self.bt_connected = False          # Bluetooth headphones state
        self.current_playlist = None
        self.play_position = 0             # seconds into the playlist

    # ------------------------------------------------------------------
    # Wi‑Fi handling (same as other devices)
    # ------------------------------------------------------------------
    def connect_wifi(self, ssid: str, bssid: str = None, rssi: int = None):
        if not bssid:
            bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        if not rssi:
            rssi = random.randint(-70, -40)

        self.connected_ssid = ssid
        self.emitter.emit_wifi_connect("ipad", ssid, bssid, rssi)
        time.sleep(_jitter() * self.speed_factor)

    def disconnect_wifi(self):
        if self.connected_ssid:
            self.emitter.emit_wifi_disconnect("ipad", self.connected_ssid)
            self.connected_ssid = None
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Bluetooth headphone pairing (used for in‑flight audio)
    # ------------------------------------------------------------------
    def bluetooth_pair(self, device_name: str = "Earbuds"):
        """
        Simulate pairing Bluetooth headphones.  Emits a ``bluetooth_pair_start``
        event, waits a short random delay, then emits ``bluetooth_pair_stop``
        when the device is “connected”.
        """
        if self.bt_connected:
            log.debug("iPad already has Bluetooth headphones paired.")
            return

        self.emitter._emit_common(
            "ipad",
            "bluetooth_pair_start",
            {"device_name": device_name}
        )
        time.sleep(_jitter(200, 800) * self.speed_factor)   # pairing latency
        self.bt_connected = True
        self.emitter._emit_common(
            "ipad",
            "bluetooth_pair_stop",
            {"device_name": device_name, "status": "connected"}
        )
        time.sleep(_jitter() * self.speed_factor)

    def bluetooth_unpair(self, device_name: str = "Earbuds"):
        """Disconnect the paired headphones."""
        if not self.bt_connected:
            return
        self.emitter._emit_common(
            "ipad",
            "bluetooth_unpair",
            {"device_name": device_name}
        )
        self.bt_connected = False
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Generic HTTP helper (used for browsing, playlist fetch, etc.)
    # ------------------------------------------------------------------
    def _http(self, method: str, url: str, **kwargs):
        headers = kwargs.pop("headers", {})
        headers["User-Agent"] = self.persona.ua["ipad"]
        if "referrer" in kwargs:
            headers["Referer"] = kwargs.pop("referrer")

        try:
            resp = requests.request(method, url, headers=headers,
                                    timeout=10, **kwargs)
            resp.raise_for_status()
            size_kb = len(resp.content) / 1024.0
            event_name = "http_get" if method.upper() == "GET" else "http_post"
            self.emitter._emit_common(
                "ipad",
                event_name,
                {"url": url, "size_kb": round(size_kb, 2)}
            )
            time.sleep(_jitter() * self.speed_factor)
            return resp
        except Exception as exc:
            log.warning(f"iPad HTTP {method} to {url} failed: {exc}")
            return None

    def http_get(self, url: str, **kwargs):
        return self._http("GET", url, **kwargs)

    def http_post(self, url: str, data=None, json_data=None, **kwargs):
        if json_data is not None:
            kwargs["json"] = json_data
        else:
            kwargs["data"] = data
        return self._http("POST", url, **kwargs)

    # ------------------------------------------------------------------
    # Music / media streaming
    # ------------------------------------------------------------------
    def load_playlist(self, playlist_id: str, title: str):
        """
        Pretend to fetch a playlist definition (list of track URLs).  The
        function stores the playlist ID locally and emits a ``playlist_load``
        event.
        """
        self.current_playlist = {
            "id": playlist_id,
            "title": title,
            # In a real implementation we'd fetch the track list; here we
            # fabricate a few placeholder URLs.
            "tracks": [
                f"https://media.example.com/track/{playlist_id}_01.mp3",
                f"https://media.example.com/track/{playlist_id}_02.mp3",
                f"https://media.example.com/track/{playlist_id}_03.mp3"
            ]
        }
        self.play_position = 0
        self.emitter._emit_common(
            "ipad",
            "playlist_load",
            {"playlist_id": playlist_id, "title": title, "track_count": len(self.current_playlist["tracks"])}
        )
        time.sleep(_jitter() * self.speed_factor)

    def stream_music(self, chunk_seconds: int = 10, max_chunks: int = 5):
        """
        Simulate streaming the currently loaded playlist.  For each chunk
        we emit a ``media_stream_chunk`` event that includes the track URL,
        chunk index, and size (approximate).

        The function respects ``self.play_position`` so consecutive calls
        continue where the previous one left off.
        """
        if not self.current_playlist:
            log.info("iPad stream_music called with no playlist loaded.")
            return

        tracks = self.current_playlist["tracks"]
        total_tracks = len(tracks)

        for _ in range(max_chunks):
            # Determine which track we are on based on play_position
            track_idx = (self.play_position // (60 * 5)) % total_tracks  # rough estimate
            track_url = tracks[track_idx]

            # Simulate a chunk download (we just issue a tiny GET)
            params = {"range": f"bytes={self.play_position}-{self.play_position + chunk_seconds * 16000 - 1}"}
            resp = self.http_get(track_url, params=params, referrer=None)

            # Emit the streaming‑chunk event
            self.emitter._emit_common(
                "ipad",
                "media_stream_chunk",
                {
                    "playlist_id": self.current_playlist["id"],
                    "track_url": track_url,
                    "chunk_index": self.play_position // (chunk_seconds * 16000),
                    "duration_sec": chunk_seconds,
                    "approx_size_kb": round(chunk_seconds * 16, 2)   # 16 KB/s approx for 128 kbps audio
                }
            )
            self.play_position += chunk_seconds
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Clipboard handling (mirrors other devices)
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copy something to the iPad’s clipboard and broadcast the sync to all
        devices via the shared Persona.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="ipad",
            content_type=content_type,
            content=content,
            description=description
        )
        for dev in ("phone", "laptop", "watch", "ipad"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "ipad",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "open_browser_tab"):
        """
        Simulate pasting the current clipboard content.
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("iPad paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "ipad",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Battery handling (iPads are relatively power‑hungry)
    # ------------------------------------------------------------------
    def drain_battery(self, amount: float):
        """
        Decrease battery level by ``amount`` percent (used during media
        streaming or heavy browsing).
        """
        self.persona.drain_battery("ipad", amount)
        self.emitter._emit_common(
            "ipad",
            "battery_update",
            {"battery_percent": self.persona.battery["ipad"]}
        )
        time.sleep(_jitter() * self.speed_factor)

    def charge_battery(self, amount: float):
        """
        Increase battery level by ``amount`` percent (e.g., when the iPad
        is placed on a charging pad in a lounge).
        """
        self.persona.charge_battery("ipad", amount)
        self.emitter._emit_common(
            "ipad",
            "battery_update",
            {"battery_percent": self.persona.battery["ipad"]}
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Simulated “conveyor‑belt” pause (iPad stays on the belt)
    # ------------------------------------------------------------------
    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        The iPad remains on the X‑ray belt: Wi‑Fi stays connected, but we
        emit a short idle heartbeat to prove the device is still powered.
        """
        for _ in range(duration_seconds):
            self.emitter._emit_common(
                "ipad",
                "idle_heartbeat",
                {"note": "iPad on conveyor belt – no user interaction"}
            )
            time.sleep(1 * self.speed_factor)

        # After the belt we emit a stabilization event (nothing actually changes)
        self.emitter._emit_common(
            "ipad",
            "device_stabilization",
            {"note": "conveyor belt passage completed"}
        )
        time.sleep(_jitter() * self.speed_factor)