# src/devices/ereader.py
"""
E‑reader (Kindle‑style) simulation.

The ``EReader`` class mimics a dedicated e‑book device that a traveler
might use to read a novel or a travel guide while waiting in an airport
lounge.  It connects to Wi‑Fi, downloads a book (simulated as a small
binary payload), syncs reading progress, and participates in the global
clipboard synchronization.

All actions are routed through the shared ``Persona`` instance so the
phone, laptop, and watch stay perfectly in sync.
"""

import logging
import random
import time
from urllib.parse import urlencode

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Tiny jitter helper – makes timing look human‑like
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class EReader:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.connected_ssid = None
        self.current_book_id = None
        self.read_progress = 0  # percentage (0‑100)

    # ------------------------------------------------------------------
    # Wi‑Fi handling (same pattern as other devices)
    # ------------------------------------------------------------------
    def connect_wifi(self, ssid: str, bssid: str = None, rssi: int = None):
        if not bssid:
            bssid = f"02:42:{random.randint(0,255):02x}:{random.randint(0,255):02x}:00:{random.randint(0,255):02x}"
        if not rssi:
            rssi = random.randint(-70, -40)

        self.connected_ssid = ssid
        self.emitter.emit_wifi_connect("ereader", ssid, bssid, rssi)
        time.sleep(_jitter() * self.speed_factor)

    def disconnect_wifi(self):
        if self.connected_ssid:
            self.emitter.emit_wifi_disconnect("ereader", self.connected_ssid)
            self.connected_ssid = None
            time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Book download (simulated)
    # ------------------------------------------------------------------
    def download_book(self, book_id: str, title: str):
        """
        Simulate fetching an e‑book file (e.g., .azw3).  The function emits
        a ``ereader_download_start`` event, performs a tiny HTTP GET (the
        URL is fabricated – we don’t actually download a large file), and
        then emits ``ereader_download_complete`` with the size.

        :param book_id: Internal identifier (e.g., ISBN or a mock UUID).
        :param title: Human‑readable title (used in logs).
        """
        self.current_book_id = book_id
        fake_url = f"https://example.com/ebooks/{book_id}.azw3"

        # ---- start event ----
        self.emitter._emit_common(
            "ereader",
            "ereader_download_start",
            {"book_id": book_id, "title": title, "url": fake_url}
        )
        time.sleep(_jitter() * self.speed_factor)

        # ---- perform a tiny GET (just to have a network request) ----
        resp = requests.get(fake_url, timeout=5)
        size_kb = len(resp.content) / 1024.0 if resp and resp.ok else 0.5  # fallback size

        # ---- complete event ----
        self.emitter._emit_common(
            "ereader",
            "ereader_download_complete",
            {
                "book_id": book_id,
                "title": title,
                "url": fake_url,
                "size_kb": round(size_kb, 2)
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Reading progress sync (called periodically while the user reads)
    # ------------------------------------------------------------------
    def sync_progress(self, percent: int):
        """
        Update the reading progress (0‑100) and emit a sync event.
        The percentage is clamped to the valid range.
        """
        percent = max(0, min(100, percent))
        self.read_progress = percent
        self.emitter._emit_common(
            "ereader",
            "ereader_sync",
            {
                "book_id": self.current_book_id,
                "progress_percent": percent
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Clipboard handling (mirrors other devices)
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copy something to the e‑reader’s clipboard and broadcast the sync
        to all devices via the shared Persona.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="ereader",
            content_type=content_type,
            content=content,
            description=description
        )
        for dev in ("phone", "laptop", "watch", "ereader"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "ereader",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "open_browser_tab"):
        """
        Simulate pasting the current clipboard content.
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("E‑Reader paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "ereader",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Battery handling (e‑readers are low‑power, but we still model it)
    # ------------------------------------------------------------------
    def drain_battery(self, amount: float):
        """
        Decrease battery level by ``amount`` percent.
        """
        self.persona.drain_battery("ereader", amount)
        self.emitter._emit_common(
            "ereader",
            "battery_update",
            {"battery_percent": self.persona.battery.get("ereader", 100)}
        )
        time.sleep(_jitter() * self.speed_factor)

    def charge_battery(self, amount: float):
        """
        Increase battery level by ``amount`` percent.
        """
        self.persona.charge_battery("ereader", amount)
        self.emitter._emit_common(
            "ereader",
            "battery_update",
            {"battery_percent": self.persona.battery.get("ereader", 100)}
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Simulated “device lost on conveyor belt” (mirrors other devices)
    # ------------------------------------------------------------------
    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        The e‑reader stays on the belt (Wi‑Fi stays connected) but the
        user isn’t interacting with it.  We emit a short “idle” heartbeat.
        """
        # No Wi‑Fi disconnect – the device stays associated
        for _ in range(duration_seconds):
            self.emitter._emit_common(
                "ereader",
                "idle_heartbeat",
                {"note": "device on conveyor belt, no interaction"}
            )
            time.sleep(1 * self.speed_factor)

        # After the belt, reconnect (no change needed because we never left)
        # but we can emit a “device_stabilization” event for completeness
        self.emitter._emit_common(
            "ereader",
            "device_stabilization",
            {"note": "conveyor belt passage completed"}
        )
        time.sleep(_jitter() * self.speed_factor)