# src/devices/braille.py
"""
Braille‑display (assistive‑technology) simulation.

This module is geared toward users who rely on a refreshable Braille
display (or a screen‑reader) to interact with their devices.  The
``Braille`` class therefore focuses on:

* **Bluetooth pairing** – the display connects to the phone (or any other
  device) via BLE.
* **Text‑to‑Braille conversion** – when the user “copies” something, the
  same content is sent to the Braille device, which then emits a
  ``braille_text_send`` event.
* **Acknowledgement** – the display sends back an ``braille_ack`` event.
* **Clipboard synchronization** – the Braille display participates in the
  same ``clipboard_sync`` flow as phone/laptop/watch.
* **Battery handling** – low‑power device, drains slowly during
  transmission.
* **Tactile‑feedback events** – optional ``braille_feedback`` events
  (e.g., a short vibration when a new line arrives).

All actions read/write the shared ``Persona`` instance, guaranteeing that
the three primary devices (phone, laptop, watch) stay perfectly in sync
with the assistive peripheral.

The implementation deliberately stays lightweight – it does **not** render
actual Braille glyphs; it merely logs the textual payload so downstream
analytics can see that a Braille device was involved.
"""

import logging
import random
import time
from typing import Optional

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Tiny jitter helper – makes timing look human‑like
# ----------------------------------------------------------------------
def _jitter(min_ms: int = 30, max_ms: int = 2000) -> float:
    """Return a random delay in seconds between min_ms and max_ms."""
    return random.uniform(min_ms, max_ms) / 1000.0


# ----------------------------------------------------------------------
class Braille:
    def __init__(self, persona, event_emitter, speed_factor=0.001):
        """
        :param persona: Shared Persona instance.
        :param event_emitter: Instance of ``src.events.EventEmitter``.
        :param speed_factor: Accelerates simulated time (real_seconds *
                             speed_factor = sleep).  Same factor used
                             throughout the toolkit.
        """
        self.persona = persona
        self.emitter = event_emitter
        self.speed_factor = speed_factor
        self.bt_connected = False          # Bluetooth‑LE connection state
        self.last_sent_text: Optional[str] = None

    # ------------------------------------------------------------------
    # Bluetooth pairing – used when the user enables the Braille display
    # ------------------------------------------------------------------
    def bluetooth_pair(self, device_name: str = "Braille_Display"):
        """
        Simulate pairing the Braille peripheral via BLE.
        Emits ``braille_pair_start`` → latency → ``braille_pair_stop``.
        """
        if self.bt_connected:
            log.debug("Braille device already paired.")
            return

        self.emitter._emit_common(
            "braille",
            "braille_pair_start",
            {"device_name": device_name}
        )
        # Simulated pairing latency (200‑800 ms)
        time.sleep(_jitter(200, 800) * self.speed_factor)
        self.bt_connected = True
        self.emitter._emit_common(
            "braille",
            "braille_pair_stop",
            {"device_name": device_name, "status": "connected"}
        )
        time.sleep(_jitter() * self.speed_factor)

    def bluetooth_unpair(self, device_name: str = "Braille_Display"):
        """Disconnect the peripheral."""
        if not self.bt_connected:
            return
        self.emitter._emit_common(
            "braille",
            "braille_unpair",
            {"device_name": device_name}
        )
        self.bt_connected = False
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Text → Braille transmission (core assistive‑tech flow)
    # ------------------------------------------------------------------
    def send_text(self, source_device: str, text: str):
        """
        Send a string to the Braille display.  The function:

        1️⃣ Emits ``braille_text_send`` (includes source device, raw text,
           and a tiny size estimate).
        2️⃣ Drains a small amount of battery (Braille displays are low‑power).
        3️⃣ Emits ``braille_ack`` after a short latency to acknowledge
           receipt.
        4️⃣ Optionally emits a short ``braille_feedback`` event (a
           vibration or click) to simulate tactile feedback.

        :param source_device: Which device originated the copy (phone,
                              laptop, watch, or braille itself).
        :param text: The textual payload to render in Braille.
        """
        if not self.bt_connected:
            log.warning("Attempted to send text to an unpaired Braille device.")
            return

        # Estimate size (1 byte per character) – round to 2 dp
        size_kb = round(len(text.encode("utf-8")) / 1024.0, 2)

        # 1️⃣ Send event
        self.emitter._emit_common(
            "braille",
            "braille_text_send",
            {
                "source_device": source_device,
                "text": text,
                "size_kb": size_kb
            }
        )
        self.last_sent_text = text

        # 2️⃣ Battery drain (very small – 0.05 % per transmission)
        self.persona.drain_battery("braille", 0.05)
        self.emitter._emit_common(
            "braille",
            "battery_update",
            {"battery_percent": self.persona.battery.get("braille", 100)}
        )
        time.sleep(_jitter() * self.speed_factor)

        # 3️⃣ ACK after a short latency (100‑300 ms)
        time.sleep(_jitter(100, 300) * self.speed_factor)
        self.emitter._emit_common(
            "braille",
            "braille_ack",
            {"status": "ok", "received_text": text[:30] + ("…" if len(text) > 30 else "")}
        )
        time.sleep(_jitter() * self.speed_factor)

        # 4️⃣ Optional tactile feedback (a brief vibration)
        self.emitter._emit_common(
            "braille",
            "braille_feedback",
            {"type": "vibration", "duration_ms": random.randint(30, 80)}
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Clipboard integration – Braille participates in the global sync
    # ------------------------------------------------------------------
    def copy_to_clipboard(self, content_type: str, content: str, description: str = ""):
        """
        Copy something to the Braille device’s clipboard and broadcast the
        sync to all other devices (phone, laptop, watch).  The Braille
        device is treated as the *source* of the copy.
        """
        entry = self.persona.copy_to_clipboard(
            source_device="braille",
            content_type=content_type,
            content=content,
            description=description
        )
        for dev in ("phone", "laptop", "watch", "braille"):
            self.emitter._emit_common(
                dev,
                "clipboard_sync",
                {
                    "source_device": "braille",
                    "content_type": content_type,
                    "content": content,
                    "description": description,
                    "synced_to": dev
                }
            )
        time.sleep(_jitter() * self.speed_factor)

    def paste_clipboard(self, action: str = "read_aloud"):
        """
        Simulate the user requesting the Braille display to read the
        current clipboard content.  The action defaults to ``read_aloud``,
        which is appropriate for a tactile device.
        """
        entry = self.persona.get_clipboard()
        if not entry:
            log.info("Braille paste requested but clipboard is empty.")
            return

        self.emitter._emit_common(
            "braille",
            "clipboard_paste",
            {
                "content_type": entry["content_type"],
                "content": entry["content"],
                "action": action
            }
        )
        # For realism, also send the text to the display (so the user can
        # feel it) – reuse ``send_text`` with source_device = "braille".
        self.send_text(source_device="braille", text=entry["content"])
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Battery handling (low‑power device)
    # ------------------------------------------------------------------
    def drain_battery(self, amount: float):
        """
        Decrease battery level by ``amount`` percent.
        """
        self.persona.drain_battery("braille", amount)
        self.emitter._emit_common(
            "braille",
            "battery_update",
            {"battery_percent": self.persona.battery.get("braille", 100)}
        )
        time.sleep(_jitter() * self.speed_factor)

    def charge_battery(self, amount: float):
        """
        Increase battery level by ``amount`` percent.
        """
        self.persona.charge_battery("braille", amount)
        self.emitter._emit_common(
            "braille",
            "battery_update",
            {"battery_percent": self.persona.battery.get("braille", 100)}
        )
        time.sleep(_jitter() * self.speed_factor)

    # ------------------------------------------------------------------
    # Simulated “device lost on conveyor belt” – Braille stays connected
    # ------------------------------------------------------------------
    def simulate_conveyor_belt(self, duration_seconds: int = 5):
        """
        When the user’s phone/laptop is placed on the X‑ray belt, the
        Braille peripheral remains paired (it has its own battery).  We
        emit a short ``idle_heartbeat`` event each second to prove the
        device is still alive.
        """
        for _ in range(duration_seconds):
            self.emitter._emit_common(
                "braille",
                "idle_heartbeat",
                {"note": "Braille device on conveyor belt – no user interaction"}
            )
            time.sleep(1 * self.speed_factor)

        # After the belt, emit a stabilization event (nothing actually
        # changes, but it mirrors the pattern used by other devices).
        self.emitter._emit_common(
            "braille",
            "device_stabilization",
            {"note": "conveyor belt passage completed"}
        )
        time.sleep(_jitter() * self.speed_factor)