# src/api_clients/wigle.py
"""
Wi‑GLE client – retrieve real Wi‑Fi SSIDs for a given airport (or any lat/lon).

The toolkit uses this module to populate the `wifi_ssids` list for each
airport in `config/airports.yaml`.  If the API call fails (rate‑limit,
network error, missing key) the function falls back to a tiny static
cache (`data/static/wifi_lists.json`) so the simulator can still run.

Required environment variables (stored in `config/api_keys.env`):
    WIGLE_KEY      – your Wi‑GLE API key (username/email)
    WIGLE_SECRET   – your Wi‑GLE API secret
"""

import os
import json
import logging
from pathlib import Path
from typing import List, Dict, Any, Optional

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Constants
# ----------------------------------------------------------------------
WIGLE_ENDPOINT = "https://api.wigle.net/api/v2/network/search"
STATIC_CACHE_PATH = Path(__file__).parents[2] / "data" / "static" / "wifi_lists.json"
DEFAULT_RADIUS_METERS = 300   # typical radius for “airport vicinity” searches


# ----------------------------------------------------------------------
# Helper – build query parameters for the Wi‑GLE request
# ----------------------------------------------------------------------
def _build_params(lat: float, lon: float, radius: int = DEFAULT_RADIUS_METERS) -> dict:
    """
    Wi‑GLE expects a rectangular bounding box.  We approximate a square
    around the centre point by adding/subtracting a small delta.
    """
    # Rough conversion: 0.001° ≈ 111 m at the equator.
    # For a 300 m radius we use ±0.003°
    delta = radius / 111_000.0   # metres → decimal degrees
    return {
        "latrange1": lat - delta,
        "latrange2": lat + delta,
        "longrange1": lon - delta,
        "longrange2": lon + delta,
        "first": 0,               # start index for pagination
        "resultsPerPage": 100,    # max per request (Wi‑GLE limit)
    }


# ----------------------------------------------------------------------
# Core API call – returns raw Wi‑GLE JSON (or None on failure)
# ----------------------------------------------------------------------
def _query_wigle(lat: float, lon: float, radius: int = DEFAULT_RADIUS_METERS) -> Optional[Dict[str, Any]]:
    api_key = os.getenv("WIGLE_KEY")
    api_secret = os.getenv("WIGLE_SECRET")
    if not api_key or not api_secret:
        log.error("WIGLE credentials missing – set WIGLE_KEY and WIGLE_SECRET in api_keys.env")
        return None

    params = _build_params(lat, lon, radius)
    try:
        resp = requests.get(
            WIGLE_ENDPOINT,
            params=params,
            auth=(api_key, api_secret),
            timeout=10,
        )
        resp.raise_for_status()
        return resp.json()
    except Exception as exc:
        log.warning(f"WIGLE request failed (lat={lat}, lon={lon}, r={radius}): {exc}")
        return None


# ----------------------------------------------------------------------
# Public helper – returns a **deduplicated list of SSIDs** for the given airport
# ----------------------------------------------------------------------
def get_wifi_ssids(iata: str,
                  lat: float,
                  lon: float,
                  radius: int = DEFAULT_RADIUS_METERS) -> List[str]:
    """
    1️⃣ Try the live Wi‑GLE API.
    2️⃣ If the API fails or returns no results, fall back to the static cache
       (which you can populate manually or let the script write to on first success).

    The returned list contains only the SSID strings (no BSSIDs, no duplicate names).

    :param iata: Airport IATA code (used only for logging / cache key)
    :param lat: Latitude of the airport centre
    :param lon: Longitude of the airport centre
    :param radius: Search radius in metres (default 300 m)
    :return: List of SSID strings
    """
    # ---- 1️⃣ Attempt live lookup ----
    raw = _query_wigle(lat, lon, radius)
    ssids: List[str] = []

    if raw and raw.get("success"):
        results = raw.get("results", [])
        ssids = list({r.get("ssid") for r in results if r.get("ssid")})
        if ssids:
            log.info(f"WIGLE: retrieved {len(ssids)} SSIDs for {iata}")
            # Cache the fresh list for future runs
            _update_static_cache(iata, ssids)
            return ssids
        else:
            log.info(f"WIGLE returned no SSIDs for {iata}")

    # ---- 2️⃣ Fallback to static cache ----
    cached = _load_from_static_cache(iata)
    if cached:
        log.info(f"Using cached Wi‑Fi list for {iata} ({len(cached)} entries)")
        return cached

    # ---- 3️⃣ If everything fails, return a minimal placeholder list ----
    placeholder = [f"Free_WiFi_{iata}", f"{iata}_Guest_WiFi"]
    log.warning(f"Returning placeholder SSIDs for {iata}: {placeholder}")
    return placeholder


# ----------------------------------------------------------------------
# Cache handling – store/retrieve per‑airport SSID lists in the JSON file
# ----------------------------------------------------------------------
def _load_from_static_cache(iata: str) -> List[str]:
    """
    Reads the static JSON cache and returns the list for the given IATA,
    or an empty list if the airport is not present.
    """
    if not STATIC_CACHE_PATH.is_file():
        return []

    try:
        with open(STATIC_CACHE_PATH, "r", encoding="utf-8") as f:
            data = json.load(f)
        return data.get(iata.upper(), [])
    except Exception as exc:
        log.warning(f"Failed to read static Wi‑Fi cache: {exc}")
        return []


def _update_static_cache(iata: str, ssid_list: List[str]) -> None:
    """
    Writes/updates the static cache file with the new SSID list for the
    given airport.  The file structure is a simple dict:
        { "LHR": ["Free_WiFi_LHR", "Delta_SkyClub_WiFi", ...], ... }
    """
    cache: Dict[str, List[str]] = {}
    if STATIC_CACHE_PATH.is_file():
        try:
            with open(STATIC_CACHE_PATH, "r", encoding="utf-8") as f:
                cache = json.load(f)
        except Exception as exc:
            log.warning(f"Could not load existing Wi‑Fi cache: {exc}")

    cache[iata.upper()] = ssid_list
    try:
        STATIC_CACHE_PATH.parent.mkdir(parents=True, exist_ok=True)
        with open(STATIC_CACHE_PATH, "w", encoding="utf-8") as f:
            json.dump(cache, f, indent=2, ensure_ascii=False)
        log.debug(f"Updated static Wi‑Fi cache for {iata}")
    except Exception as exc:
        log.error(f"Failed to write static Wi‑Fi cache: {exc}")How to use it in the toolkitfrom src.api_clients.wigle import get_wifi_ssids

# Example inside src/itinerary.py when building an airport record:
airport = airports_cfg["LHR"]
wifi_list = get_wifi_ssids(
    iata=airport["airport_id"],
    lat=airport["lat"],
    lon=airport["lon"]
)
airport["wifi_ssids"] = wifi_list
