# src/api_clients/wettr.py
"""
Simple wrapper around the `wettr.in` service.

The toolkit uses this to fetch the **current temperature** for a city.
`wettr.in` returns a short string like "12°C" or "55°F".  We parse the
numeric value and the unit so the rest of the code can work with a
float temperature and a canonical unit identifier.

No API key is required – the service is public.
"""

import logging
import re
from typing import Optional, Tuple

import requests

log = logging.getLogger(__name__)

WETTR_BASE_URL = "https://wettr.in"


# ----------------------------------------------------------------------
def _parse_response(text: str) -> Optional[Tuple[float, str]]:
    """
    Expected format from wettr.in:
        "12°C"   → 12.0, "C"
        "55°F"   → 55.0, "F"
    The function extracts the numeric part and the trailing letter.
    """
    match = re.search(r"([+-]?\d+(?:\.\d*)?)\s*°?\s*([CF])", text.strip(), re.IGNORECASE)
    if not match:
        log.warning(f"Unable to parse wettr.in response: '{text}'")
        return None
    value = float(match.group(1))
    unit = match.group(2).upper()
    return value, unit


# ----------------------------------------------------------------------
def get_temperature(city: str, unit: str = "C") -> Optional[float]:
    """
    Query wettr.in for the current temperature in the given city.

    :param city: City name (e.g. "London", "New York").
    :param unit: Desired output unit – "C" for Celsius or "F" for Fahrenheit.
                 The request uses the `u=` query param to ask wettr.in for the
                 correct unit; if omitted it defaults to Celsius.
    :return: Temperature as a float (in the requested unit) or None on error.
    """
    # Build the request URL
    # Example: https://wettr.in/London?format=%t&u=c
    fmt_param = "%t"  # only the temperature value
    params = {"format": fmt_param}
    if unit.upper() == "F":
        params["u"] = "f"

    url = f"{WETTR_BASE_URL}/{city}"
    try:
        resp = requests.get(url, params=params, timeout=5)
        resp.raise_for_status()
        # wettr returns something like "12°C" or "55°F"
        parsed = _parse_response(resp.text)
        if parsed:
            temperature, _ = parsed
            return temperature
        else:
            return None
    except Exception as exc:
        log.warning(f"wettr.in request failed for city '{city}': {exc}")
        return None


# ----------------------------------------------------------------------
# Convenience helper – returns a tuple (value, unit) so callers can decide
# ----------------------------------------------------------------------
def get_temperature_with_unit(city: str) -> Optional[Tuple[float, str]]:
    """
    Returns both the numeric temperature and the unit that wettr.in chose.
    Useful when you want to keep the original unit (e.g., for logging).

    :param city: City name.
    :return: (value, unit) or None.
    """
    url = f"{WETTR_BASE_URL}/{city}?format=%t"
    try:
        resp = requests.get(url, timeout=5)
        resp.raise_for_status()
        parsed = _parse_response(resp.text)
        return parsed
    except Exception as exc:
        log.warning(f"wettr.in request failed for city '{city}': {exc}")
        return None