# src/api_clients/tripadvisor.py
"""
TripAdvisor (via RapidAPI) – fetch “things to do” / tourist attractions.

The toolkit uses this to obtain a **real‑world attraction URL** that can be
copied to the clipboard (e.g., “Eiffel Tower” → https://www.tripadvisor.com/…).

You need a RapidAPI subscription for the TripAdvisor API.  Add the key to
`config/api_keys.env`:

    RAPIDAPI_TRIPADVISOR_KEY=YOUR_RAPIDAPI_KEY

If the key is missing or the request fails, the function falls back to a
tiny static list (defined at the bottom of the file) so the simulator
continues to run without interruption.
"""

import os
import json
import logging
from pathlib import Path
from typing import List, Dict, Any, Optional

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Endpoint & static fallback
# ----------------------------------------------------------------------
RAPIDAPI_HOST = "tripadvisor1.p.rapidapi.com"
RAPIDAPI_ENDPOINT = f"https://{RAPIDAPI_HOST}/places/object_search"

# Path to a tiny static JSON fallback (you can extend it if you like)
STATIC_FALLBACK_PATH = Path(__file__).parents[2] / "data" / "static" / "tripadvisor_fallback.json"


# ----------------------------------------------------------------------
# Helper – build request headers for RapidAPI
# ----------------------------------------------------------------------
def _build_headers() -> Dict[str, str]:
    api_key = os.getenv("RAPIDAPI_TRIPADVISOR_KEY")
    if not api_key:
        log.error("RapidAPI TripAdvisor key not set – add RAPIDAPI_TRIPADVISOR_KEY to api_keys.env")
        return {}
    return {
        "x-rapidapi-key": api_key,
        "x-rapidapi-host": RAPIDAPI_HOST,
        "useQueryString": "true"
    }


# ----------------------------------------------------------------------
# Core request – returns raw JSON from the TripAdvisor API
# ----------------------------------------------------------------------
def _query_tripadvisor(city: str, limit: int = 5) -> Optional[Dict[str, Any]]:
    """
    Perform a free‑text search for attractions in the given city.

    :param city: City name (e.g. "Paris").
    :param limit: Maximum number of results to request.
    :return: Parsed JSON dict or None on failure.
    """
    headers = _build_headers()
    if not headers:
        return None

    params = {
        "query": city,
        "lang": "en_US",
        "limit": limit,
    }

    try:
        resp = requests.get(RAPIDAPI_ENDPOINT, headers=headers, params=params, timeout=10)
        resp.raise_for_status()
        return resp.json()
    except Exception as exc:
        log.warning(f"TripAdvisor request failed for city '{city}': {exc}")
        return None


# ----------------------------------------------------------------------
# Public helper – returns a list of attraction dicts
# ----------------------------------------------------------------------
def get_attractions(city: str, limit: int = 5) -> List[Dict[str, Any]]:
    """
    Retrieve a short list of popular attractions for a city.

    Each entry in the returned list contains:
        - name: Human‑readable attraction name
        - url:  Direct TripAdvisor URL (or a placeholder if unavailable)
        - rating: Average rating (float, 0‑5) – may be None if not present
        - description: Short snippet (optional)

    If the API call fails, the function loads a tiny static fallback file
    (`data/static/tripadvisor_fallback.json`).  The fallback contains a
    generic entry for every city you have in `destinations.yaml`; you can
    edit that file to add more attractions.

    :param city: City name.
    :param limit: Max number of attractions to return.
    :return: List of dictionaries.
    """
    raw = _query_tripadvisor(city, limit=limit)
    attractions: List[Dict[str, Any]] = []

    if raw and raw.get("data"):
        # The API returns a list under the "data" key; each item has a
        # "result_object" with the details we need.
        for item in raw["data"][:limit]:
            obj = item.get("result_object", {})
            name = obj.get("name")
            url = obj.get("web_url")
            rating = obj.get("rating")  # may be a string like "4.5"
            desc = obj.get("description")
            if name and url:
                attractions.append({
                    "name": name,
                    "url": url,
                    "rating": float(rating) if rating else None,
                    "description": desc or ""
                })
        if attractions:
            log.info(f"TripAdvisor: found {len(attractions)} attractions for {city}")
            return attractions

    # ------------------------------------------------------------------
    # Fallback – load static JSON if the live request gave nothing
    # ------------------------------------------------------------------
    log.info(f"Using TripAdvisor fallback data for city '{city}'")
    try:
        with open(STATIC_FALLBACK_PATH, "r", encoding="utf-8") as f:
            fallback_data = json.load(f)
        city_key = city.lower()
        if city_key in fallback_data:
            return fallback_data[city_key][:limit]
    except Exception as exc:
        log.warning(f"Failed to load TripAdvisor fallback file: {exc}")

    # If even the fallback is missing, return a generic placeholder
    placeholder = [{
        "name": f"Popular spot in {city}",
        "url": f"https://www.tripadvisor.com/Search?q={city.replace(' ', '+')}",
        "rating": None,
        "description": ""
    }]
    return placeholder