# src/api_clients/osm.py
"""
OpenStreetMap helpers – routing (OSRM) and geocoding (Nominatim).

The toolkit uses these functions to:
  1️⃣ Generate a realistic driving route when the clone “asks the taxi where
     to go”.
  2️⃣ Convert a human‑readable destination (e.g. “Hotel Le Meurice”) into
     latitude/longitude so the route can be calculated.

Both services are public and do **not** require an API key, but they enforce
rate‑limits.  The functions include a tiny back‑off and cache layer to stay
friendly to the providers.

Typical usage:

    # Geocode a destination string
    lat, lon = geocode_address("Hotel Le Meurice, Paris")

    # Get a driving route from the airport to that point
    route = get_route(
        origin_lat=airport_lat,
        origin_lon=airport_lon,
        dest_lat=lat,
        dest_lon=lon,
    )
"""

import logging
import time
from typing import Tuple, Dict, Any, Optional

import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Configuration
# ----------------------------------------------------------------------
OSRM_ENDPOINT = "https://router.project-osrm.org/route/v1/driving"
NOMINATIM_ENDPOINT = "https://nominatim.openstreetmap.org/search"

# Simple in‑memory cache for geocoding results (address → (lat, lon))
_GEOCODE_CACHE: Dict[str, Tuple[float, float]] = {}

# ----------------------------------------------------------------------
# Helper – exponential back‑off decorator (used for both services)
# ----------------------------------------------------------------------
def _retry(max_attempts: int = 3, backoff: float = 1.0):
    def decorator(func):
        def wrapper(*args, **kwargs):
            delay = backoff
            for attempt in range(1, max_attempts + 1):
                try:
                    return func(*args, **kwargs)
                except Exception as exc:
                    if attempt == max_attempts:
                        log.error(f"{func.__name__} failed after {attempt} attempts: {exc}")
                        raise
                    log.warning(f"{func.__name__} attempt {attempt} failed: {exc} – retrying in {delay}s")
                    time.sleep(delay)
                    delay *= 2  # exponential back‑off
        return wrapper
    return decorator

# ----------------------------------------------------------------------
# Geocoding – turn a free‑form address into (lat, lon)
# ----------------------------------------------------------------------
@_retry(max_attempts=3, backoff=1.0)
def geocode_address(query: str, limit: int = 1) -> Tuple[float, float]:
    """
    Resolve a human‑readable address using Nominatim.

    :param query: Address string (e.g. "Hotel Le Meurice, Paris").
    :param limit: Number of results to request (default 1 – we keep the best match).
    :return: (latitude, longitude) as floats.
    :raises: RuntimeError if no result is found.
    """
    # Check the simple cache first
    cache_key = query.lower().strip()
    if cache_key in _GEOCODE_CACHE:
        return _GEOCODE_CACHE[cache_key]

    params = {
        "q": query,
        "format": "json",
        "limit": limit,
        "addressdetails": 0,
    }
    headers = {
        "User-Agent": "NoiseOrchestra/1.0 (+https://github.com/yourrepo/noise-orchestra)"
    }

    resp = requests.get(NOMINATIM_ENDPOINT, params=params, headers=headers, timeout=8)
    resp.raise_for_status()
    results = resp.json()

    if not results:
        raise RuntimeError(f"Nominatim could not resolve address: '{query}'")

    # Take the first result
    first = results[0]
    lat = float(first["lat"])
    lon = float(first["lon"])

    # Store in cache for future calls
    _GEOCODE_CACHE[cache_key] = (lat, lon)
    log.debug(f"Geocoded '{query}' → ({lat}, {lon})")
    return lat, lon

# ----------------------------------------------------------------------
# Routing – OSRM driving directions
# ----------------------------------------------------------------------
@_retry(max_attempts=3, backoff=1.0)
def get_route(origin_lat: float,
              origin_lon: float,
              dest_lat: float,
              dest_lon: float,
              overview: str = "full",
              geometries: str = "polyline",
              steps: bool = True) -> Dict[str, Any]:
    """
    Query the public OSRM service for a car route.

    :param origin_lat: Latitude of the start point.
    :param origin_lon: Longitude of the start point.
    :param dest_lat: Latitude of the destination.
    :param dest_lon: Longitude of the destination.
    :param overview: Level of geometry detail ("simplified", "full", "false").
    :param geometries: Geometry encoding ("polyline", "geojson").
    :param steps: Include turn‑by‑turn instructions (True/False).
    :return: Dictionary with the most useful fields:
        {
            "distance": meters (float),
            "duration": seconds (float),
            "polyline": encoded geometry string,
            "steps": [ { "instruction": "...", "distance": ..., "duration": ... }, ... ]
        }
    :raises: RuntimeError if OSRM returns an error or no routes.
    """
    # Build the coordinate string: lon,lat;lon,lat (OSRM expects lng,lat)
    coord_str = f"{origin_lon},{origin_lat};{dest_lon},{dest_lat}"
    url = f"{OSRM_ENDPOINT}/{coord_str}"

    params = {
        "overview": overview,
        "geometries": geometries,
        "steps": "true" if steps else "false",
        "alternatives": "false"
    }

    resp = requests.get(url, params=params, timeout=10)
    resp.raise_for_status()
    data = resp.json()

    if data.get("code") != "Ok" or not data.get("routes"):
        raise RuntimeError(f"OSRM routing error: {data.get('message', 'unknown')}")

    route = data["routes"][0]   # we asked for a single best route
    result: Dict[str, Any] = {
        "distance": route.get("distance"),          # metres
        "duration": route.get("duration"),          # seconds
        "polyline": route.get("geometry") if geometries == "polyline" else None,
        "steps": []
    }

    if steps:
        # Flatten the step list across all legs (normally just one leg)
        for leg in route.get("legs", []):
            for step in leg.get("steps", []):
                instr = step.get("maneuver", {}).get("instruction", "")
                result["steps"].append({
                    "instruction": instr,
                    "distance": step.get("distance"),
                    "duration": step.get("duration")
                })

    log.debug(
        f"OSRM route: {origin_lat},{origin_lon} → {dest_lat},{dest_lon} "
        f"→ {result['distance']:.1f} m, {result['duration']:.1f} s"
    )
    return result

# ----------------------------------------------------------------------
# Convenience wrapper – one‑liner for the typical “taxi” use‑case
# ----------------------------------------------------------------------
def get_taxi_route(origin: Tuple[float, float],
                   destination_query: str) -> Dict[str, Any]:
    """
    High‑level helper used by the itinerary when the clone asks the taxi
    “where would you like to go?”.  It:

    1️⃣ Geocodes the free‑form destination string.
    2️⃣ Calls OSRM to get a driving route from the origin point.
    3️⃣ Returns the OSRM payload (distance, duration, steps, polyline).

    :param origin: (lat, lon) tuple for the taxi’s starting location.
    :param destination_query: Human‑readable address (e.g. “Hotel Le Meurice, Paris”).
    :return: Same structure as `get_route`.
    """
    dest_lat, dest_lon = geocode_address(destination_query)
    return get_route(
        origin_lat=origin[0],
        origin_lon=origin[1],
        dest_lat=dest_lat,
        dest_lon=dest_lon,
        overview="full",
        geometries="polyline",
        steps=True
    )