# src/api_clients/aviationstack.py
"""
AviationStack API client – fetch real‑time flight information.

The toolkit uses this module in `src/itinerary.py` to obtain a live
flight for the selected origin → destination pair.  The function
`get_flight_info` returns a **single** flight dictionary (or ``None`` if no
matching flight is found).

Environment variable required:
    AVIATIONSTACK_KEY – your AviationStack API key.
"""

import os
import logging
from typing import Optional, Dict, Any
import requests

log = logging.getLogger(__name__)

# ----------------------------------------------------------------------
# Helper – build the request URL & parameters
# ----------------------------------------------------------------------
def _build_params(dep_iata: str, arr_iata: str, flight_date: str) -> dict:
    """
    :param dep_iata: IATA code of the departure airport (e.g. "LHR")
    :param arr_iata: IATA code of the arrival airport (e.g. "JFK")
    :param flight_date: ISO‑date string (YYYY‑MM‑DD) – the day you want to fly
    :return: dict of query parameters for the AviationStack endpoint
    """
    return {
        "access_key": os.getenv("AVIATIONSTACK_KEY"),
        "dep_iata": dep_iata,
        "arr_iata": arr_iata,
        "flight_date": flight_date,          # AviationStack expects YYYY‑MM‑DD
        "limit": 50,                         # keep the payload small
    }

# ----------------------------------------------------------------------
# Public API – fetch a single flight (the first that matches our criteria)
# ----------------------------------------------------------------------
def get_flight_info(dep_iata: str,
                    arr_iata: str,
                    date: str) -> Optional[Dict[str, Any]]:
    """
    Retrieve a real‑time flight record from AviationStack.

    The function attempts to find **any** scheduled flight that departs from
    ``dep_iata`` to ``arr_iata`` on the given ``date``.  If multiple flights
    are returned we simply pick the first one (the toolkit can be extended
    to apply additional weighting).

    Returned dictionary (example keys):
        - flight_number      (e.g. "AA1234")
        - airline            (marketing airline name)
        - operating_airline  (IATA code of the carrier actually operating the flight)
        - aircraft_type      (e.g. "B777")
        - scheduled_departure (ISO‑8601 string, local time)
        - actual_departure    (ISO‑8601 string, may be None if not departed)
        - delay_minutes       (int, may be 0)
        - gate                (string, may be None)
        - status              (e.g. "active", "landed", "cancelled")
        - status_url          (link to the airline’s flight‑status page, if provided)

    If the API key is missing or the request fails, ``None`` is returned and a
    warning is logged.

    :param dep_iata: Departure airport IATA code.
    :param arr_iata: Arrival airport IATA code.
    :param date: Desired flight date as ``YYYY-MM-DD``.
    :return: Dictionary with flight details or ``None``.
    """
    api_key = os.getenv("AVIATIONSTACK_KEY")
    if not api_key:
        log.error("AviationStack API key not set – please add AVIATIONSTACK_KEY to config/api_keys.env")
        return None

    endpoint = "http://api.aviationstack.com/v1/flights"
    params = _build_params(dep_iata, arr_iata, date)

    try:
        response = requests.get(endpoint, params=params, timeout=10)
        response.raise_for_status()
        data = response.json()
    except Exception as exc:
        log.warning(f"AviationStack request failed: {exc}")
        return None

    # The API returns a list under the "data" key
    flights = data.get("data", [])
    if not flights:
        log.info(f"No flights found on {date} from {dep_iata} → {arr_iata}")
        return None

    # Pick the first flight that has a scheduled departure time
    for fl in flights:
        # Basic sanity checks – we need at least a flight number and a scheduled time
        if not fl.get("flight_number") or not fl.get("departure", {}).get("scheduled"):
            continue

        # Build a clean dict for the toolkit
        flight_info = {
            "flight_number": fl.get("flight_number"),
            "airline": fl.get("airline", {}).get("name"),
            "operating_airline_iata": fl.get("airline", {}).get("iata_code"),
            "aircraft_type": fl.get("aircraft", {}).get("registration") or fl.get("aircraft", {}).get("model"),
            "scheduled_departure": fl.get("departure", {}).get("scheduled"),
            "actual_departure": fl.get("departure", {}).get("actual"),
            "delay_minutes": fl.get("departure", {}).get("delay") or 0,
            "gate": fl.get("departure", {}).get("gate"),
            "status": fl.get("flight_status"),
            # Some airlines expose a direct status URL; if not, we construct a generic one
            "status_url": fl.get("flight_status_url") or f"https://www.{fl.get('airline',{}).get('name','').lower().replace(' ','')}.com/flightstatus/{fl.get('flight_number')}"
        }
        return flight_info

    # If we fell through the loop, no suitable flight was found
    log.info(f"No suitable flight record after filtering for {dep_iata}->{arr_iata} on {date}")
    return None