/**
 * stealth_browser.js
 * -----------------
 * Headless‑browser “human” traffic generator.
 *
 *   • Reads runtime options from config.json (targets, proxy, UA list, etc.).
 *   • Picks a random target URL, random UA, random language, random viewport.
 *   • Optionally routes traffic through an HTTP(S) proxy.
 *   • Navigates, waits a few seconds (simulating a real user), then closes.
 *   • Prints ONE line to STDOUT:
 *         [BROW] visited https://example.com/ (headless)
 *
 * The orchestrator captures that line, prefixes it with a timestamp, and
 * writes it to noise_test.log.
 */

const fs   = require('fs');
const path = require('path');
const puppeteer = require('puppeteer-extra');
const StealthPlugin = require('puppeteer-extra-plugin-stealth');

// ------------------------------------------------------------
// 1️⃣  Load configuration (fallbacks if fields are missing)
// ------------------------------------------------------------
const CONFIG_PATH = path.join(__dirname, 'config.json');
let rawConfig = {};

try {
  rawConfig = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf-8'));
} catch (e) {
  console.error('[BROW] ❌ Could not read config.json – using defaults.');
}

// Helper to pick a random element from an array
const randChoice = arr => arr[Math.floor(Math.random() * arr.length)];

// Targets – must be an array of URLs; fall back to a tiny default list
const TARGETS = Array.isArray(rawConfig.targets) && rawConfig.targets.length
  ? rawConfig.targets
  : ['https://www.google.com/', 'https://www.wikipedia.org/'];

// Proxy configuration
const PROXY_ENABLED = rawConfig.proxy?.enabled ?? false;
const PROXY_URL     = rawConfig.proxy?.url ?? '';

// User‑Agent pool
const USER_AGENTS = Array.isArray(rawConfig.userAgents) && rawConfig.userAgents.length
  ? rawConfig.userAgents
  : [
      'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36',
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 13_5) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.5 Safari/605.1.15',
      'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36',
      'Mozilla/5.0 (iPhone; CPU iPhone OS 17_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.0 Mobile/15E148 Safari/604.1',
      'Mozilla/5.0 (Linux; Android 13; SM-G991U) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Mobile Safari/537.36'
    ];

// Accept‑Language header pool
const LANGUAGES = Array.isArray(rawConfig.languages) && rawConfig.languages.length
  ? rawConfig.languages
  : ['en-US,en;q=0.9', 'fr-FR,fr;q=0.8', 'de-DE,de;q=0.7'];

// Viewport sizes (width × height)
const RESOLUTIONS = Array.isArray(rawConfig.screenResolutions) && rawConfig.screenResolutions.length
  ? rawConfig.screenResolutions
  : [{ width: 1366, height: 768 }, { width: 1920, height: 1080 }, { width: 1440, height: 900 }];

// How long to stay on a page (seconds)
const MIN_VISIT = Number.isFinite(rawConfig.minVisitTimeSec) ? rawConfig.minVisitTimeSec : 5;
const MAX_VISIT = Number.isFinite(rawConfig.maxVisitTimeSec) ? rawConfig.maxVisitTimeSec : 12;

// ------------------------------------------------------------
// 2️⃣  Puppeteer setup (Stealth plugin + optional proxy)
// ------------------------------------------------------------
puppeteer.use(StealthPlugin());

(async () => {
  // Random selections for this run
  const targetUrl = randChoice(TARGETS);
  const userAgent = randChoice(USER_AGENTS);
  const acceptLang = randChoice(LANGUAGES);
  const viewport   = randChoice(RESOLUTIONS);
  const visitTimeMs = (Math.random() * (MAX_VISIT - MIN_VISIT) + MIN_VISIT) * 1000;

  // Build launch arguments
  const launchOpts = {
    headless: true,
    args: [
      '--no-sandbox',
      '--disable-setuid-sandbox',
      '--disable-dev-shm-usage',
      '--disable-gpu',
      `--window-size=${viewport.width},${viewport.height}`
    ],
    defaultViewport: {
      width: viewport.width,
      height: viewport.height,
    },
  };

  if (PROXY_ENABLED && PROXY_URL) {
    launchOpts.args.push(`--proxy-server=${PROXY_URL}`);
  }

  let browser;
  try {
    browser = await puppeteer.launch(launchOpts);
    const page = await browser.newPage();

    // --------------------------------------------------------
    // Compatibility shim – make sure we have a reliable delay method
    // --------------------------------------------------------
    // 1️⃣ If the native waitForTimeout exists – great.
    // 2️⃣ If only the old waitFor exists – map waitForTimeout → waitFor.
    // 3️⃣ If neither exists – implement a tiny promise‑based timeout.
    if (typeof page.waitForTimeout !== 'function') {
      if (typeof page.waitFor === 'function') {
        // Old Puppeteer (< v3) – reuse its waitFor
        page.waitForTimeout = (ms) => page.waitFor(ms);
      } else {
        // Newer Puppeteer where both helpers are missing (rare with puppeteer‑extra)
        // Simple setTimeout‑based promise:
        page.waitForTimeout = (ms) => new Promise(resolve => setTimeout(resolve, ms));
      }
    }

    // (Optional) sanity‑check – you can uncomment to see which branch was taken
    // console.log('has waitForTimeout?', typeof page.waitForTimeout);

    // Apply random UA, language, and other headers
    await page.setUserAgent(userAgent);
    await page.setExtraHTTPHeaders({
      'Accept-Language': acceptLang,
    });

    // Hide the language list from fingerprinting (Stealth already does most of this)
    await page.evaluateOnNewDocument((lang) => {
      Object.defineProperty(navigator, 'languages', {
        get: () => [lang.split(',')[0]],
      });
    }, acceptLang);

    // Navigate to the target URL
    await page.goto(targetUrl, { waitUntil: 'domcontentloaded', timeout: 30000 });

    // Stay on the page for a random amount of time (simulates reading)
    await page.waitForTimeout(visitTimeMs);

    // OPTIONAL: a tiny scroll to look more human‑like
    await page.evaluate(() => {
      window.scrollBy(0, Math.floor(window.innerHeight / 2));
    });

    // Clean up
    await page.close();
    await browser.close();

    // Emit the one‑line log entry that the orchestrator will capture
    console.log(`[BROW] visited ${targetUrl} (headless)`);
  } catch (err) {
    // Even on failure we want a line so the orchestrator knows something ran
    console.error(`[BROW] ❌ error visiting ${targetUrl}: ${err.message}`);
    if (browser) {
      try { await browser.close(); } catch (_) {}
    }
    process.exit(1);
  }
})();