#!/usr/bin/env python3
"""
Noise Orchestra – Test Run (1-hour)

* Starts the same sub-processes you already have (web flood, IoT, browser bot,
  telemetry noise).
* Instead of sending each sub-process to /dev/null, we capture its stdout
  and prepend a UTC timestamp.
* The script stops automatically after TEST_DURATION seconds (default = 3600s).

When you’re satisfied, run the same file with `--live` – it will detach
the children and let them run forever (or you can schedule it with cron).
"""

import argparse, subprocess, sys, time, datetime, os, signal
from pathlib import Path

# ----------------------------------------------------------------------
# Configuration
# ----------------------------------------------------------------------
TEST_DURATION = 3600                # seconds → 1hour (change as you wish)
LOGFILE = Path(__file__).with_name("noise_test.log")

# ----------------------------------------------------------------------
# Helper: write a line with a UTC timestamp
# ----------------------------------------------------------------------
def log_line(prefix: str, msg: str) -> None:
    ts = datetime.datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S.%f")[:-3]
    line = f"[{ts}] [{prefix}] {msg}"
    print(line)                     # also goes to console for quick watching
    with LOGFILE.open("a", encoding="utf-8") as f:
        f.write(line + "\n")

# ----------------------------------------------------------------------
# Helper: launch a subprocess and forward its stdout line-by-line
# ----------------------------------------------------------------------
def launch(name: str, cmd: list):
    """Start a child process, capture its stdout, and log each line."""
    proc = subprocess.Popen(
        cmd,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT,
        text=True,
        bufsize=1,                  # line-buffered
    )
    # Forward each line as it arrives
    def forward():
        for line in proc.stdout:   # iterates until EOF
            clean = line.rstrip()
            if clean:
                log_line(name, clean)
    # Run the forwarder in a lightweight thread
    import threading
    t = threading.Thread(target=forward, daemon=True)
    t.start()
    return proc, t

# ----------------------------------------------------------------------
# Main orchestration
# ----------------------------------------------------------------------
def run_test():
    LOGFILE.touch(exist_ok=True)   # create/empty the log file
    log_line("MAIN", f"Starting ONE-HOUR test (duration={TEST_DURATION}s)")

    # ------------------------------------------------------------------
    # 1️⃣ Core noise components (same binaries you already have)
    # ------------------------------------------------------------------
    children = []
    children.append(launch("WEB",   ["python3", "noise_flood.py"]))
    children.append(launch("IOT",   ["python3", "iot_simulator.py"]))
    children.append(launch("BROW",  ["node", "stealth_browser.js"]))
    children.append(launch("TEL",   ["python3", "telemetry_noise.py"]))

    # ------------------------------------------------------------------
    # 2️⃣ Extra “light” helpers for the test run only
    # ------------------------------------------------------------------
    # BLE beacon spam – we just log a fake beacon line every ~10 s
    ble_script = """
import time, random, sys
while True:
    sys.stdout.write("[BLE] beacon UUID=%s\\n" % ''.join(random.choice('0123456789abcdef') for _ in range(32)))
    sys.stdout.flush()
    time.sleep(random.uniform(8,12))
"""
    children.append(launch("BLE", ["python3", "-c", ble_script]))

    # Wi-Fi probe logger – write a line every 30-90 s
    wifi_script = """
import time, random, sys
ssids = ["CoffeeShop_WiFi","Airport_Free","Library_Public","Starbucks_1234"]
while True:
    sys.stdout.write("[WIFI] probe SSID=%s\\n" % random.choice(ssids))
    sys.stdout.flush()
    time.sleep(random.uniform(30,90))
"""
    children.append(launch("WIFI", ["python3", "-c", wifi_script]))

    # DNS query logger – fire a dig every 2-5 min (output suppressed, we just log)
    dns_script = """
import time, random, subprocess, sys
domains = ["example.com","mozilla.org","github.com","wikipedia.org"]
while True:
    d = random.choice(domains)
    subprocess.run(["dig","+short",d], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
    sys.stdout.write("[DNS] queried %s\\n" % d)
    sys.stdout.flush()
    time.sleep(random.uniform(120,300))
"""
    children.append(launch("DNS", ["python3", "-c", dns_script]))

    # ------------------------------------------------------------------
    # Run for the configured duration
    # ------------------------------------------------------------------
    end_time = time.time() + TEST_DURATION
    try:
        while time.time() < end_time:
            time.sleep(1)          # main thread just idles
    except KeyboardInterrupt:
        log_line("MAIN", "Interrupted by user – shutting down early")
    finally:
        log_line("MAIN", "Terminating all child processes")
        for proc, _ in children:
            proc.terminate()
        # Give them a moment to exit cleanly
        time.sleep(2)
        for proc, _ in children:
            if proc.poll() is None:   # still alive?
                proc.kill()
        log_line("MAIN", "Test run finished – see noise_test.log for details")

# ----------------------------------------------------------------------
# Live mode – detach and let everything run forever (or be managed by cron)
# ----------------------------------------------------------------------
def run_live():
    LOGFILE.touch(exist_ok=True)
    log_line("MAIN", "Launching LIVE orchestra (background)")

    # Same launch calls as above but without the forwarding threads;
    # we simply redirect each child's stdout to the log file.
    def launch_detached(name, cmd):
        with LOGFILE.open("a", encoding="utf-8") as f:
            # Prepend a prefix to each line via a tiny wrapper script
            wrapper = f"""
import sys, subprocess, datetime
proc = subprocess.Popen({cmd!r}, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1)
for line in proc.stdout:
    ts = datetime.datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S.%f')[:-3]
    sys.stdout.write(f'[{ts}] [{name}] {{line.rstrip()}}\\n')
    sys.stdout.flush()
"""
            subprocess.Popen(
                [sys.executable, "-c", wrapper],
                stdout=f,
                stderr=f,
                close_fds=True,
            )

    launch_detached("WEB",   ["python3", "noise_flood.py"])
    launch_detached("IOT",   ["python3", "iot_simulator.py"])
    launch_detached("BROW",  ["node", "stealth_browser.js"])
    launch_detached("TEL",   ["python3", "telemetry_noise.py"])
    # (optional) add BLE/Wi-Fi/DNS here as well if you want them live

    log_line("MAIN", "All components started – exiting now (they keep running).")

# ----------------------------------------------------------------------
# Entry point
# ----------------------------------------------------------------------
if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Noise Orchestra – test run (default 1 hour) or live background mode."
    )
    parser.add_argument(
        "--live",
        action="store_true",
        help="Run in live/background mode (no automatic stop).",
    )
    args = parser.parse_args()

    if args.live:
        run_live()
    else:
        run_test()