#!/usr/bin/env python3
"""
noise_flood.py
----------------
Generates a short burst of HTTP GET requests to a random selection of URLs.
Designed to be launched by noise_orchestra_test.py (or run standalone).

Features
--------
* Reads every *.txt file in the script directory that looks like a URL list.
  Typical files: fluff.txt, chum.txt, news_urls.txt, weather_urls.txt …
* Optional User‑Agent rotation (ua.txt). Falls back to a small built‑in list.
* Uses Tor (SOCKS5 @ 127.0.0.1:9050) if the proxy is reachable;
  otherwise falls back to a direct connection.
* Configurable jitter, request count, and optional bandwidth cap.
* Prints one line per successful request:
      GET https://example.com/ → 200 OK
  The orchestrator prefixes each line with a timestamp and the “WEB” tag.
"""

import os
import random
import socket
import sys
import time
from pathlib import Path
from urllib.parse import urlparse

import requests

# ------------------------------------------------------------
# 1️⃣  CONFIGURATION (tweak as needed)
# ------------------------------------------------------------
# How many requests per burst (you can lower this if you have a tight data cap)
REQ_COUNT = 12

# Minimum / maximum seconds between individual requests inside a burst
INTERVAL_RANGE = (5, 15)          # seconds – adds jitter

# Optional bandwidth limit per burst (bytes). Set to None to disable.
# Example: 500_000  → ~0.5 MB per burst (well under a 5 % monthly cap for most plans)
MAX_BYTES_PER_BURST = 500_000

# Tor SOCKS5 proxy address – change only if you run Tor on a different port
TOR_SOCKS = ("127.0.0.1", 9050)

# ------------------------------------------------------------
# 2️⃣  HELPERS – loading files, picking random items, etc.
# ------------------------------------------------------------
SCRIPT_DIR = Path(__file__).parent.resolve()


def load_url_pool() -> list[str]:
    """
    Scan the script directory for any *.txt file that looks like a URL list.
    Returns a flat list of stripped, non‑empty lines.
    """
    url_files = sorted([p for p in SCRIPT_DIR.iterdir()
                       if p.is_file() and p.suffix.lower() == ".txt"
                       and p.name not in {"process_names.txt",
                                          "dns_domains.txt",
                                          "sensor_names.txt",
                                          "ua.txt"}])  # exclude non‑URL txts

    urls: list[str] = []
    for fp in url_files:
        for line in fp.read_text(encoding="utf-8").splitlines():
            line = line.strip()
            if line and not line.startswith("#"):
                urls.append(line)
    if not urls:
        sys.stderr.write("❌ No URLs found – please add at least one URL to a .txt file.\n")
        sys.exit(1)
    return urls


def load_user_agents() -> list[str]:
    """Read ua.txt if present; otherwise return a small built‑in list."""
    ua_path = SCRIPT_DIR / "ua.txt"
    if ua_path.is_file():
        uas = [ln.strip() for ln in ua_path.read_text(encoding="utf-8")
               .splitlines() if ln.strip()]
        if uas:
            return uas

    # Fallback list (covers Windows, macOS, Linux, iOS, Android)
    return [
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 13_5) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.5 Safari/605.1.15",
        "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36",
        "Mozilla/5.0 (iPhone; CPU iPhone OS 17_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.0 Mobile/15E148 Safari/604.1",
        "Mozilla/5.0 (Linux; Android 13; SM-G991U) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Mobile Safari/537.36",
    ]


def tor_is_available() -> bool:
    """Quick check – try opening a TCP connection to the Tor SOCKS port."""
    try:
        with socket.create_connection(TOR_SOCKS, timeout=1):
            return True
    except OSError:
        return False


def build_session(use_tor: bool) -> requests.Session:
    """Create a requests Session with optional Tor SOCKS proxy."""
    sess = requests.Session()
    # Reasonable timeout – we don’t want a single slow site to stall the burst
    sess.headers.update({"Accept": "*/*", "Connection": "close"})
    if use_tor:
        socks_proxy = f"socks5h://{TOR_SOCKS[0]}:{TOR_SOCKS[1]}"
        sess.proxies.update({"http": socks_proxy, "https": socks_proxy})
    return sess


def size_of_response(resp: requests.Response) -> int:
    """Return an approximate byte size of the response (headers + body)."""
    hdr_len = sum(len(k) + len(v) + 4 for k, v in resp.headers.items())  # ': ' + '\r\n'
    body_len = len(resp.content) if resp.content else 0
    return hdr_len + body_len


# ------------------------------------------------------------
# 3️⃣  MAIN BURST LOGIC
# ------------------------------------------------------------
def run_burst(urls: list[str], uas: list[str]) -> None:
    """
    Perform a single burst of REQ_COUNT requests.
    Prints a line for each successful request.
    """
    use_tor = tor_is_available()
    sess = build_se