#!/usr/bin/env python3
"""
iot_simulator.py
----------------
Light‑weight MQTT “IoT device farm” simulator.

Features
--------
* Optional device catalogue (devices.json). If absent, a small built‑in list is used.
* Optional MQTT config (mqtt_config.yaml). If absent, defaults to the public
  Mosquitto test broker (test.mosquitto.org:1883, no auth).
* Each device publishes a JSON payload on its own topic:
      iot/<device_id>/state
* Payload fields are chosen to match the device type (temperature, humidity,
  motion, battery, lock_state, etc.).
* Randomised publish interval per device (30 s – 180 s) to avoid a regular pattern.
* Writes a concise line to stdout for every successful publish:
      [IOT] thermostat-01 → {"temp":22.4,"hum":45}
  The orchestrator captures this line and adds its own timestamp/tag.
* Automatic reconnection logic and clean shutdown on SIGINT/SIGTERM.
"""

import json
import random
import signal
import sys
import time
from pathlib import Path
from typing import Dict, List

import paho.mqtt.client as mqtt
import yaml

# ------------------------------------------------------------
# 1️⃣  CONFIGURATION (tweak if you need something else)
# ------------------------------------------------------------
SCRIPT_DIR = Path(__file__).parent.resolve()

# Default MQTT broker (public test broker – no auth)
DEFAULT_MQTT = {
    "host": "test.mosquitto.org",
    "port": 1883,
    "username": None,
    "password": None,
    "keepalive": 60,
}

# Publish interval range per device (seconds)
INTERVAL_RANGE = (30, 180)   # 0.5 min – 3 min

# ------------------------------------------------------------
# 2️⃣  HELPERS – loading JSON/YAML files, building payloads
# ------------------------------------------------------------
def load_devices() -> List[Dict]:
    """
    Load device definitions from devices.json if it exists.
    Expected format:
    [
        {"id": "thermostat-01", "type": "temperature", "unit": "°C"},
        {"id": "motion-01",    "type": "motion"},
        ...
    ]
    If the file is missing or malformed, fall back to a built‑in list.
    """
    dev_path = SCRIPT_DIR / "devices.json"
    if dev_path.is_file():
        try:
            data = json.loads(dev_path.read_text(encoding="utf-8"))
            if isinstance(data, list) and data:
                return data
        except Exception as exc:
            sys.stderr.write(f"⚠️  Failed to parse devices.json: {exc}\n")

    # Built‑in fallback list (six diverse devices)
    return [
        {"id": "thermostat-01", "type": "temperature", "unit": "°C"},
        {"id": "humidity-01",   "type": "humidity",    "unit": "%"},
        {"id": "motion-01",     "type": "motion"},
        {"id": "doorlock-01",   "type": "lock"},
        {"id": "battery-01",    "type": "battery"},
        {"id": "airquality-01", "type": "aqi"},
    ]


def load_mqtt_cfg() -> Dict:
    """
    Load MQTT connection settings from mqtt_config.yaml.
    Expected keys: host, port, username, password, keepalive (optional).
    Missing file → use DEFAULT_MQTT.
    """
    cfg_path = SCRIPT_DIR / "mqtt_config.yaml"
    if cfg_path.is_file():
        try:
            cfg = yaml.safe_load(cfg_path.read_text(encoding="utf-8"))
            if isinstance(cfg, dict):
                # Merge with defaults for any missing key
                merged = DEFAULT_MQTT.copy()
                merged.update({k: v for k, v in cfg.items() if v is not None})
                return merged
        except Exception as exc:
            sys.stderr.write(f"⚠️  Failed to parse mqtt_config.yaml: {exc}\n")
    return DEFAULT_MQTT.copy()


def synth_payload(device: Dict) -> Dict:
    """
    Produce a JSON‑serialisable dict that matches the device type.
    The values are random but stay in realistic ranges.
    """
    typ = device["type"]
    if typ == "temperature":
        return {"temp": round(random.uniform(18.0, 27.0), 1)}               # °C
    if typ == "humidity":
        return {"hum": random.randint(30, 70)}                           # %
    if typ == "motion":
        return {"motion": random.choice([True, False])}
    if typ == "lock":
        return {"locked": random.choice([True, False])}
    if typ == "battery":
        return {"pct": random.randint(20, 100)}                          # %
    if typ == "aqi":
        # Air Quality Index – 0 (good) to 200 (unhealthy)
        return {"aqi": random.randint(0, 150)}
    # Generic fallback – just send a timestamp
    return {"ts": int(time.time())}


# ------------------------------------------------------------
# 3️⃣  MQTT CLIENT SETUP & RECONNECT LOGIC
# ------------------------------------------------------------
class MQTTClient:
    def __init__(self, cfg: Dict):
        self.cfg = cfg
        self.client = mqtt.Client()
        if cfg["username"]:
            self.client.username_pw_set(cfg["username"], cfg["password"])
        self.client.on_connect = self._on_connect
        self.client.on_disconnect = self._on_disconnect
        self.connected = False

    def _on_connect(self, client, userdata, flags, rc):
        if rc == 0:
            self.connected = True
        else:
            sys.stderr.write(f"⚠️  MQTT connect failed (rc={rc})\n")

    def _on_disconnect(self, client, userdata, rc):
        self.connected = False
        # rc != 0 means unexpected disconnect – we’ll try to reconnect later
        if rc != 0:
            sys.stderr.write("⚠️  Unexpected MQTT disconnect – will retry.\n")

    def connect(self):
        self.client.connect(
            self.cfg["host"],
            self.cfg.get("port", 1883),
            self.cfg.get("keepalive", 60),
        )
        self.client.loop_start()
        # Wait a short moment for the CONNECT packet to succeed
        timeout = time.time() + 5
        while not self.connected and time.time() < timeout:
            time.sleep(0.1)
        if not self.connected:
            sys.stderr.write("❌ Could not establish MQTT connection.\n")
            sys.exit(1)

    def publish(self, topic: str, payload: Dict):
        """Publish JSON payload; returns True on success."""
        if not self.connected:
            return False
        try:
            self.client.publish(topic, json.dumps(payload), qos=0)
            return True
        except Exception as exc:
            sys.stderr.write(f"⚠️  MQTT publish error: {exc}\n")
            return False

    def disconnect(self):
        self.client.loop_stop()
        self.client.disconnect()


# ------------------------------------------------------------
# 4️⃣  MAIN LOOP – per‑device publishing with jitter
# ------------------------------------------------------------
shutdown_requested = False


def handle_signal(signum, frame):
    global shutdown_requested
    shutdown_requested = True


def main():
    # Register SIGINT/SIGTERM so we can exit cleanly
    signal.signal(signal.SIGINT, handle_signal)
    signal.signal(signal.SIGTERM, handle_signal)

    devices = load_devices()
    mqtt_cfg = load_mqtt_cfg()
    mqtt_cli = MQTTClient(mqtt_cfg)
    mqtt_cli.connect()

    # Track next scheduled publish time per device
    next_pub: Dict[str, float] = {
        dev["id"]: time.time() + random.uniform(*INTERVAL_RANGE) for dev in devices
    }

    while not shutdown_requested:
        now = time.time()
        for dev in devices:
            dev_id = dev["id"]
            if now >= next_pub[dev_id]:
                payload = synth_payload(dev)
                topic = f"iot/{dev_id}/state"
                success = mqtt_cli.publish(topic, payload)

                # Output a line for the orchestrator (stdout is line‑buffered)
                status = "OK" if success else "FAIL"
                print(f"[IOT] {dev_id} → {json.dumps(payload)} ({status})")

                # Schedule next publish for this device
                next_pub[dev_id] = now + random.uniform(*INTERVAL_RANGE)

        # Sleep a short while to avoid busy‑waiting
        time.sleep(0.5)

    # Clean shutdown
    mqtt_cli.disconnect()
    print("[IOT] Simulator stopped.")


if __name__ == "__main__":
    # Ensure stdout is line‑buffered so the orchestrator sees each line immediately
    sys.stdout.reconfigure(line_buffering=True)
    main()